<?php

namespace App\Http\Controllers;

use App\Mail\RegisterUser;
use App\Models\Audit;
use App\Models\Department;
use App\Models\ItemStatus;
use App\Models\Notifications;
use App\Models\Permission;
use App\Models\Role;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx\Rels;

class SystemManagementController extends Controller
{
    public function showUserAccessControl(Request $request)
    {
        if(Auth::check()){

            $filterName = $request->input('name');
            $filterUsername = $request->input('username');
            $filterRole = $request->input('role');
            $filterActive = $request->input('active');
            $filterDepartment = $request->input('department');

            $query = DB::table('users')
                    ->join('user_role', 'ur_id','=','user_role_id')
                    ->join('department', 'department_id','=','user_department_id');

            if ($filterName) {
                $query->where('name', 'LIKE', '%'.$filterName.'%');
            }
            if ($filterUsername) {
                $query->where('username', 'LIKE', '%'.$filterUsername.'%');
            }
            if ($filterRole) {
                $query->where('ur_id', $filterRole);
            }
            if ($filterActive) {
                $query->where('is_active', $filterActive);
            }
            if($filterDepartment){
                $query->where('user_department_id', $filterDepartment);
            }

            $user = $query->get();

            $nameFilter = $query->pluck('name');
            $usernameFilter = $query->pluck('username');

            $role = Role::where('is_deleted', 0)
                    ->orderBy('ur_name', 'ASC')
                    ->get();

            $department = Department::where('is_deleted', 0)
                        ->orderBy('department_name', 'ASC')
                        ->get();

            $notification = Notifications::orderBy('notification_is_read', 'ASC')->get();
            $new_notification = Notifications::where('notification_is_read', 0)->count();

            return view('useraccesscontrol')->with([
                'user' => $user,
                'role' => $role,
                'notification' => $notification,
                'new_notification' => $new_notification,
                'filter_name' => $nameFilter,
                'filter_username' => $usernameFilter,
                'department' => $department
            ]);
        }
        return redirect()->to('login')->withSuccess('You are not allowed to access');

    }

    public function showModulePageSetup()
    {
        if(Auth::check()){

            $filter = Role::where('is_deleted', 0)
                    ->orderBy('ur_name', 'ASC')
                    ->get();

            $module = Permission::all();

            $notification = Notifications::orderBy('notification_is_read', 'ASC')->get();
            $new_notification = Notifications::where('notification_is_read', 0)->count();

            return view('modulesetup')->with([
                'filter' => $filter,
                'module' => $module,
                'notification' => $notification,
                'new_notification' => $new_notification,
            ]);
        }
        return redirect()->to('login')->withSuccess('You are not allowed to access');

    }

    public function showRole()
    {
        if(Auth::check()){

            $role = Role::where('is_deleted', 0)->orderBy('ur_id', 'DESC')->get();
            $notification = Notifications::orderBy('notification_is_read', 'ASC')->get();
            $new_notification = Notifications::where('notification_is_read', 0)->count();

            return view('role')->with([
                'role' => $role,
                'notification' => $notification,
                'new_notification' => $new_notification,
            ]);
        }
        return redirect()->to('login')->withSuccess('You are not allowed to access');

    }

    public function addRole(Request $request)
    {
        $existingName = Role::where('ur_name', $request['name'])
                        ->where('is_deleted', 0)
                        ->exists();
        $existingINameDeleted = Role::where('ur_name', $request['name'])
                            ->where('is_deleted', 1)
                            ->exists();

        if ($existingName) {
            return redirect()->back()->with('error_name1', "Role name ".$request['name']." already exists.");
        }
        if ($existingINameDeleted) {
            return redirect()->back()->with('error_name2', "Role name ".$request['name']." already exists in restore module.");
        }

        $role = new Role();
        $role->ur_name = $request['name'];
        $role->save();

        // $module = DB::table('module')->get();
        $submodule = ['Supply Management','Spare Part','Stock In/Out','Budgeting','Department','User Access Control','Module Page Setup','Item Status','Audit Trail','Restore'];

        foreach($submodule as $submodules){
            $permission = new Permission();
            $permission->up_role_id = $role->ur_id;
            $permission->up_module = $submodules;
            $permission->is_view = 0;
            $permission->is_create = 0;
            $permission->is_update= 0;
            $permission->is_delete = 0;
            $permission->save();
        }

        // foreach($module as $modules){
        //     $permission = new Permission();
        //     $permission->up_role_id = $role->ur_id;
        //     $permission->up_module = $modules->module_name;
        //     $permission->is_view = 0;
        //     $permission->is_create = 0;
        //     $permission->is_update= 0;
        //     $permission->is_delete = 0;
        //     $permission->save();
        // }

        $audit = new Audit();
        $audit->at_module = 'System Management';
        $audit->at_submodule = 'Module Page Setup';
        $audit->at_activity = 'Add Role';
        $audit->at_description = 'Add Role: '. $request['name'];
        $audit->at_user_id = Auth::user()->id;
        $audit->save();

        return redirect()->route('role');
    }

    public function addUser(Request $request)
    {
        $register = User::where('username', $request['username'])
                    ->orwhere('email', $request['email'])
                    ->orwhere('ic_no', $request['ic_no'])
                    ->count();

        if($register != 0){
            return back()->with('register','Email / UserName / IC No already exist');
        }

        $user = new User();
        $user->name = $request['name'];
        $user->username = $request['username'];
        $user->email = $request['email'];
        $user->ic_no = $request['ic_no'];
        $user->password = bcrypt($request['password']);
        $user->user_role_id = $request['role'];
        $user->user_department_id = $request['department'];
        $user->is_active = 1;
        $user->save();

        Mail::to($request['email'])->send(new RegisterUser($request['name'], $request['username'], $request['password']));

        $audit = new Audit();
        $audit->at_module = 'System Management';
        $audit->at_submodule = 'User Access Control';
        $audit->at_activity = 'Add User';
        $audit->at_description = 'Add User: '. $request['name'];
        $audit->at_user_id = Auth::user()->id;
        $audit->save();

        return redirect()->route('usercontrol');
    }

    public function showEditUser($id)
    {
        if(Auth::check()){

            $user = DB::table('users')
                    ->join('user_role', 'ur_id','=','user_role_id')
                    ->join('department', 'department_id','=','user_department_id')
                    ->where('users.id', $id)
                    ->first();

            $role = Role::orderBy('ur_name', 'ASC')->get();

            $department = Department::where('is_deleted', 0)->orderBy('department_name', 'ASC')->get();

            $notification = Notifications::orderBy('notification_is_read', 'ASC')->get();
            $new_notification = Notifications::where('notification_is_read', 0)->count();

            return view('edit-user')->with([
                'user' => $user,
                'role' => $role,
                'notification' => $notification,
                'new_notification' => $new_notification,
                'department' => $department
            ]);
        }
        return redirect()->to('login')->withSuccess('You are not allowed to access');

    }

    public function updateUser(Request $request, $id)
    {
        $user = User::find($id);

        $originalName = $user->name;
        $originalUsername = $user->username;
        $originalIC = $user->ic_no;
        $originalEmail = $user->email;
        $originalRole = Role::where('ur_id', $user->user_role_id)->value('ur_name');
        $NewRole = Role::where('ur_id', $request['role'])->value('ur_name');
        $originalDepartment = Department::where('department_id', $user->user_department_id)->value('department_name');
        $NewDepartment = Department::where('department_id', $request['department'])->value('department_name');
        $originalActive = $user->is_active;

        $user->name = $request->input('name');
        $user->username = $request->input('username');
        $user->ic_no = $request->input('icno');
        $user->email = $request->input('email');
        $user->user_role_id = $request->input('role');
        $user->user_department_id = $request->input('department');
        $user->is_active = ($request->has('is_active') ? 1 : 0);

        $changes = [];

        // Compare and record changes
        if ($originalName !== $request['name']) {
            $changes[] = "<br>Name changed from '$originalName' to '$request[name]'";
        }
        if ($originalUsername !== $request['username']) {
            $changes[] = "<br>User Name changed from '$originalUsername' to '$request[username]'";
        }
        if ($originalIC !== $request['icno']) {
            $changes[] = "<br>IC No changed from '$originalIC' to '$request[icno]'";
        }
        if ($originalEmail !== $request['email']) {
            $changes[] = "<br>Email changed from '$originalEmail' to '$request[email]'";
        }
        if ($originalRole !== $NewRole) {
            $changes[] = "<br>Role changed from '$originalRole' to '$NewRole'";
        }
        if ($originalDepartment !== $NewDepartment) {
            $changes[] = "<br>Department changed from '$originalDepartment' to '$NewDepartment'";
        }
        $originalActiveName = ($originalActive == 1 ? 'Active' : 'Non-Active');
        $NewActiveName = $request->has('is_active') ? 1 : 0 == (1 ? 'Active' : 'Non-Active');
        if ($originalActive !== ($request->has('is_active') ? 1 : 0)) {
            $changes[] = "<br>Active User changed from '$originalActiveName ' to '$NewActiveName'";
        }

        // Create the audit description by joining the changes
        $auditDescription = 'Edited User: <br>
                        Name: '.$originalName.'
                        ' . implode(', ', $changes);

        if($changes != null){
            $audit = new Audit();
            $audit->at_module = 'System Management';
            $audit->at_submodule = 'User Access Control';
            $audit->at_activity = 'Edit User';
            $audit->at_description = $auditDescription;
            $audit->at_user_id = Auth::user()->id;
            $audit->save();
        }

        $user->save();

        return redirect()->route('usercontrol');
    }

    public function deleteUser($id)
    {
        $user = User::find($id);

        $audit = new Audit();
        $audit->at_module = 'System Management';
        $audit->at_submodule = 'User Access Control';
        $audit->at_activity = 'Delete User';
        $audit->at_description = 'Deleted User: '. $user->name;
        $audit->at_user_id = Auth::user()->id;
        $audit->save();

        $user->delete();

        return redirect()->route('usercontrol');

    }

    public function editPermission(Request $request)
    {
        $permissionsData = $request->input('permissions');

        foreach ($permissionsData as $roleId => $permission) {

            $updated = Permission::updateOrCreate(
                ['up_id' => $roleId],
                [
                    'is_view' => $permission['is_view'],
                    'is_create' => $permission['is_create'],
                    'is_update' => $permission['is_update'],
                    'is_delete' => $permission['is_delete'],
                    ]
                );
        }

        $audit = new Audit();
        $audit->at_module = 'System Management';
        $audit->at_submodule = 'Module Page Setup';
        $audit->at_activity = 'Edit Module Page Setup';
        $audit->at_description = 'Edit Module Page Setup';
        $audit->at_user_id = Auth::user()->id;
        $audit->save();

        return redirect()->back();
    }


    public function updateRole(Request $request, $ur_id)
    {
        $existingName = Role::where('ur_name', $request['name'])
                        ->where('is_deleted', 0)
                        ->exists();
        $existingINameDeleted = Role::where('ur_name', $request['name'])
                            ->where('is_deleted', 1)
                            ->exists();

        if ($existingName) {
            return redirect()->back()->with('error_name1', "Role name ".$request['name']." already exists.");
        }
        if ($existingINameDeleted) {
            return redirect()->back()->with('error_name2', "Role name ".$request['name']." already exists in restore module.");
        }

        $role = Role::find($ur_id);

        $originalName = $role->ur_name;

        $role->ur_name = $request['name'];

        $changes = [];

        // Compare and record changes
        if ($originalName !== $request['name']) {
            $changes[] = "<br>Name changed from '$originalName' to '$request[name]'";
        }

        // Create the audit description by joining the changes
        $auditDescription = 'Edited Role: ' . implode(', ', $changes);

        if($changes != null){
            $audit = new Audit();
            $audit->at_module = 'System Management';
            $audit->at_submodule = 'Role';
            $audit->at_activity = 'Edit Role';
            $audit->at_description = $auditDescription;
            $audit->at_user_id = Auth::user()->id;
            $audit->save();
        }

        $role->save();

        return redirect()->back();

    }

    public function deleteRole($ur_id)
    {
        $role = Role::find($ur_id);

        $user = User::where('user_role_id', $role->ur_id)->count();

        if($user == 0){
            $audit = new Audit();
            $audit->at_module = 'System Management';
            $audit->at_submodule = 'Role';
            $audit->at_activity = 'Delete Role';
            $audit->at_description = 'Deleted Role:'. $role->ur_name;
            $audit->at_user_id = Auth::user()->id;
            $audit->save();

            //Delete user permission based on role id

            $role->is_deleted = 1;
            $role->save();

        }else{
            return back()->withErrors(['user' => 'error']);
        }

        return redirect()->back();

    }

    public function restoreRole($ur_id)
    {
        $role = Role::find($ur_id);
        $role->is_deleted = 0;
        $role->save();

        $audit = new Audit();
        $audit->at_module = 'Restore';
        $audit->at_submodule = 'Restore';
        $audit->at_activity = 'Restore Role';
        $audit->at_description = 'Restore Role: '.$role->ur_name;
        $audit->at_user_id = Auth::user()->id;
        $audit->save();

        return redirect()->route('restore');
    }

    public function permanentDeleteRole($ur_id)
    {
        $role = Role::find($ur_id);

        $audit = new Audit();
        $audit->at_module = 'Restore';
        $audit->at_submodule = 'Restore';
        $audit->at_activity = 'Delete Role';
        $audit->at_description = 'Permanently Delete Role: '.$role->ur_name;
        $audit->at_user_id = Auth::user()->id;
        $audit->save();

        $role->delete();

        return redirect()->back();
    }

    public function showItemStatus()
    {
        if(Auth::check()){

            $department = Department::where('is_deleted', 0)->orderBy('department_id', 'DESC')->get();

            $status = ItemStatus::get();
            $notification = Notifications::orderBy('notification_is_read', 'ASC')->get();
            $new_notification = Notifications::where('notification_is_read', 0)->count();

            return view('itemstatus')->with([
                'department' => $department,

                'status' => $status,
                'notification' => $notification,
                'new_notification' => $new_notification,
            ]);
        }
        return redirect()->to('login')->withSuccess('You are not allowed to access');
    }

    public function updateItemStatus(Request $request, $is_id)
    {
        $item_status = ItemStatus::find($is_id);

        $originalStatus = $item_status->is_time;
        $item_status->is_time = $request['time'];

        $changes = [];

        // Compare and record changes
        if ($originalStatus !== $request['time']) {
            $changes[] = "<br>Status Months changed from '$originalStatus' to '$request[time]'";
        }

        // Create the audit description by joining the changes
        $auditDescription = 'Edited Item Status: ' . implode(', ', $changes);

        if($changes != null){
            $audit = new Audit();
            $audit->at_module = 'System Management';
            $audit->at_submodule = 'Item Status';
            $audit->at_activity = 'Edit Status';
            $audit->at_description = $auditDescription;
            $audit->at_user_id = Auth::user()->id;
            $audit->save();
        }

        $item_status->save();

        return redirect()->back();

    }

}
