<?php

namespace App\Http\Controllers;

use App\Models\Audit;
use App\Models\AuditDesc;
use App\Models\Notifications;
use App\Models\SparePart;
use App\Models\StockInOut;
use App\Models\Supplier;
use Carbon\Carbon;
use DateTime;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

class SupplyController extends Controller
{
    public function showSupply(Request $request)
    {
        if(Auth::check()){

            $dropFilter = Supplier::where('is_deleted', 0)->orderBy('supplier_id', 'DESC');

            $oldCodeFilter = $dropFilter->where('supplier_oldcode', '!=', null)->pluck('supplier_oldcode')->unique();
            $newCodeFilter = $dropFilter->where('supplier_newcode', '!=', null)->pluck('supplier_newcode')->unique();
            $supplierFilter = $dropFilter->where('supplier_name', '!=', null)->pluck('supplier_name')->unique();
            $controlFilter = $dropFilter->where('supplier_control_acc', '!=', null)->pluck('supplier_control_acc')->unique();
            $vendorFilter = $dropFilter->where('supplier_vendor_group', '!=', null)->pluck('supplier_vendor_group')->unique();

            $filterOldCode = $request->input('old_code');
            $filterNewCode = $request->input('new_code');
            $filterSupplierName = $request->input('supplier_name');
            $filterControlAcc = $request->input('control_acc');
            $filterVendor = $request->input('vendor');

            $query = Supplier::query();

             // Apply filters
            if ($filterOldCode) {
                $query->where('supplier_oldcode', $filterOldCode);
            }

            if ($filterNewCode) {
                $query->where('supplier_newcode', $filterNewCode);
            }

            if ($filterSupplierName) {
                $query->where('supplier_name', 'LIKE', '%'.$filterSupplierName.'%');
            }

            if ($filterControlAcc) {
                $query->where('supplier_control_acc', $filterControlAcc);
            }

            if ($filterVendor) {
                $query->where('supplier_vendor_group','LIKE', '%'.$filterVendor.'%');
            }

            $supplier = $query->where('is_deleted', 0)->orderBy('supplier_id', 'DESC')->get();
            $total = $query->count();

            // $supplier = Supplier::where('is_deleted', 0)->orderBy('supplier_id', 'DESC')->get();
            // $total = Supplier::where('is_deleted', 0)->count();
            $notification = Notifications::orderBy('notification_is_read', 'ASC')->get();
            $new_notification = Notifications::where('notification_is_read', 0)->count();

            return view('supply')->with([
                'supplier' => $supplier,
                'total' => $total,
                'notification' => $notification,
                'new_notification' => $new_notification,
                'filter_oldcode' => $oldCodeFilter,
                'filter_newcode' => $newCodeFilter,
                'filter_supplier' => $supplierFilter,
                'filter_control' => $controlFilter,
                'filter_vendor' => $vendorFilter
            ]);
        }

        return redirect()->to('login')->withSuccess('You are not allowed to access');
    }

    public function addSupplierIndividual(Request $request)
    {
        $existingSupplier = Supplier::where('supplier_newcode', $request['new_code'])->where('is_deleted', 0)->count();
        $existingSupplier2 = Supplier::where('supplier_newcode', $request['new_code'])->where('is_deleted', 1)->count();

        if ($existingSupplier != 0) {
            return back()->with('error', 'Supplier new code already exists.');
        }
        if ($existingSupplier2 != 0) {
            return back()->with('error2', 'Supplier new code already exists in restore module.');
        }

        $supply = new Supplier();
        $supply->supplier_oldcode = $request['old_code'];
        $supply->supplier_newcode = $request['new_code'];
        $supply->supplier_name = $request['name'];
        $supply->supplier_control_acc = $request['acc'];
        $supply->supplier_vendor_group = $request['vendor'];
        $supply->save();

        $audit = new Audit();
        $audit->at_module = 'Supply Management';
        $audit->at_submodule = 'Supply Management';
        $audit->at_activity = 'Add Supplier by Individual';
        $audit->at_description = 'Add Supplier: '.$supply->supplier_name
                                .'<br>New code: '.$supply->supplier_newcode
                                .'<br>Vendor: '.$supply->supplier_vendor_group;
        $audit->at_user_id = Auth::user()->id;
        $audit->save();

        return redirect()->route('supply');
    }

    public function addSupplierBulk(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx',
        ]);

        $file = $request->file('file');
        $fileName = $file->getClientOriginalName();

        $data = Excel::toArray([], $file);

        // Skip the first header in excel
        $data = array_slice($data[0], 1);

        $audit = new Audit();
        $audit->at_module = 'Supply Management';
        $audit->at_submodule = 'Supply Management';
        $audit->at_activity = 'Add Supplier by Group';
        $audit->at_description = null;
        $audit->at_user_id = Auth::user()->id;
        $audit->save();

        $no = 1;

        foreach ($data as $row) {
            $supplierNewCode = $row[1];

            // Check if supplier_newcode already exists in the database
            if (Supplier::where('supplier_newcode', $supplierNewCode)->exists()) {
                // return redirect()->back()->with('error','Supplier new code already exists.');
                $duplicateSupplierNewCodes[] = $supplierNewCode;
                continue;
            }

            // If the supplier_newcode doesn't exist, insert the data
            Supplier::create([
                'supplier_oldcode' => $row[0],
                'supplier_newcode' => $row[1],
                'supplier_name' => $row[2],
                'supplier_control_acc' => $row[3],
                'supplier_vendor_group' => $row[4],
            ]);

            $auditDescription = 'Add Suppliers by Group in excel ('.$no.')';
            $auditDescription .= '<br>Supplier name: ' . $row[2];
            $auditDescription .= '<br>New Code: ' . $row[1];
            $auditDescription .= '<br>Vendor: ' . $row[4];

            $audit_desc = new AuditDesc();
            $audit_desc->ad_audit_id = $audit->at_id;
            $audit_desc->ad_description = $auditDescription;
            $audit_desc->save();

            $no++;
        }

        // Check if there are duplicate codes and generate an error message
        if (!empty($duplicateSupplierNewCodes)) {
            $errorMessage = 'The following Supplier New Code values already exist: ' . implode(', ', $duplicateSupplierNewCodes);
            return redirect()->back()->with('error', $errorMessage);
        }


        return redirect()->back()->with('success', 'Data imported successfully!');

    }

    public function downloadTemplate()
    {
        $file_path = public_path('template/Add_Supplier_Bulk.xlsx');
        $headers = [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ];
        return response()->download($file_path, 'Add_Supplier_Bulk.xlsx', $headers);
    }

    public function deleteSupplier($supplier_id)
    {

        $supplier = Supplier::where('supplier_id', $supplier_id)->first();
        $supplier->is_deleted = 1;
        $supplier->save();

        $part = SparePart::where('sp_supplier_id', $supplier_id)->get();
        if($part->count() != 0){
            foreach($part as $parts){
                $parts->is_deleted = 1;
                $parts->save();
            }

            $stock = StockInOut::where('si_sp_id', $part->sp_id)->get();
            if($stock->count() != 0){
                foreach($stock as $stocks){
                    $stocks->is_deleted = 1;
                    $stocks->save();
                }
            }
        }

        $audit = new Audit();
        $audit->at_module = 'Supply Management';
        $audit->at_submodule = 'Supply Management';
        $audit->at_activity = 'Delete Supplier';
        $audit->at_description = 'Deleted Supplier Name <br>
                                  Supplier name: '. $supplier->supplier_name.'<br>
                                  New code: '.$supplier->newcode;

        $audit->at_user_id = Auth::user()->id;
        $audit->save();

        return redirect()->back();
    }

    public function showEditSupplier($supplier_id)
    {
        if(Auth::check()){

            $supplier = Supplier::where('supplier_id', $supplier_id)->first();
            $notification = Notifications::orderBy('notification_is_read', 'ASC')->get();
            $new_notification = Notifications::where('notification_is_read', 0)->count();

            return view('edit-supply')->with([
                'supplier' => $supplier,
                'notification' => $notification,
                'new_notification' => $new_notification,
            ]);
        }

        return redirect()->to('login')->withSuccess('You are not allowed to access');
    }

    public function updateSupplier(Request $request, $supplier_id)
    {
        $supply = Supplier::find($supplier_id);

        $originalOldCode = $supply->supplier_oldcode;
        $originalNewCode = $supply->supplier_newcode;
        $originalName = $supply->supplier_name;
        $originalControl = $supply->supplier_control_acc;
        $originalVendor = $supply->supplier_vendor_group;

        // $existingSupplier = Supplier::where('supplier_newcode', $request['new_code'])->where('is_deleted', 0)->count();
        // if($existingSupplier != 0){
        //     return back()->with('error', 'New Code already existed');
        // }

        $supply->supplier_oldcode = $request['old_code'];
        $supply->supplier_newcode = $request['new_code'];
        $supply->supplier_name = $request['name'];
        $supply->supplier_control_acc = $request['acc'];
        $supply->supplier_vendor_group = $request['vendor'];

        $changes = [];

        // Compare and record changes
        if ($originalOldCode !== $request['old_code']) {
            $changes[] = "<br>Old Code changed from '$originalOldCode' to '$request[old_code]'";
        }
        if ($originalNewCode !== $request['new_code']) {
            $changes[] = "<br>New Code changed from '$originalNewCode' to '$request[new_code]'";
        }
        if ($originalName !== $request['name']) {
            $changes[] = "<br>Name changed from '$originalName' to '$request[name]'";
        }
        if ($originalControl !== $request['acc']) {
            $changes[] = "<br>Control Account changed from '$originalControl' to '$request[acc]'";
        }
        if ($originalVendor !== $request['vendor']) {
            $changes[] = "<br>Vendor Group changed from '$originalVendor' to '$request[vendor]'";
        }

        // Create the audit description by joining the changes
        $auditDescription = 'Edited Supplier: <br>
                            New Code: '.$originalNewCode.'
                            ' . implode(', ', $changes);

        if($changes != null){
            $audit = new Audit();
            $audit->at_module = 'Supply Management';
            $audit->at_submodule = 'Supply Management';
            $audit->at_activity = 'Edit Supplier';
            $audit->at_description = $auditDescription;
            $audit->at_user_id = Auth::user()->id;
            $audit->save();
        }

        $supply->save();

        return redirect()->route('supply');
    }

    public function restoreSupplier($supplier_id)
    {
        $supply = Supplier::find($supplier_id);
        $supply->is_deleted = 0;
        $supply->save();

        $audit = new Audit();
        $audit->at_module = 'Restore';
        $audit->at_submodule = 'Restore';
        $audit->at_activity = 'Restore Supplier';
        $audit->at_description = 'Restore Supplier <br>
                                  Supplier name: '.$supply->supplier_name.'<br>
                                  New code: '.$supply->supplier_newcode;
        $audit->at_user_id = Auth::user()->id;
        $audit->save();

        return redirect()->route('restore');
    }

    public function permanentDeleteSupplier($supplier_id)
    {
        $supply = Supplier::find($supplier_id);

        $audit = new Audit();
        $audit->at_module = 'Restore';
        $audit->at_submodule = 'Restore';
        $audit->at_activity = 'Delete Supplier';
        $audit->at_description = 'Permanently Delete Supplier: '.$supply->supplier_name;
        $audit->at_user_id = Auth::user()->id;
        $audit->save();

        $supply->delete();

        return redirect()->back();
    }

    public function downloadAllSupplier()
    {
        $supplier = Supplier::select('supplier_oldcode','supplier_newcode','supplier_name','supplier_control_acc','supplier_vendor_group')
                ->get();

        //Get Excel Template
        $spreadsheet = IOFactory::load(public_path('template/Add_Supplier_Bulk.xlsx'));
        $sheet = $spreadsheet->getActiveSheet();

        $row = 2;
        foreach($supplier as $data){
            $sheet->setCellValue('A'.$row, $data->supplier_oldcode);
            $sheet->setCellValue('B'.$row, $data->supplier_newcode);
            $sheet->setCellValue('C'.$row, $data->supplier_name);
            $sheet->setCellValue('D'.$row, $data->supplier_control_acc);
            $sheet->setCellValue('E'.$row, $data->supplier_vendor_group);

            $row++;
        }

        $path = 'report/Supplier_report.xlsx';
        $filePath = public_path($path);

        $filenameExcel = 'Supplier_report.xlsx';

        $writer = new Xlsx($spreadsheet);
        $writer->save(public_path($path));

        return response()->download($filePath, $filenameExcel);

    }
}
