<?php

namespace App\Http\Controllers;

use App\Exports\InventoryExport;
use App\Mail\LowStock;
use App\Models\Audit;
use App\Models\AuditDesc;
use App\Models\Budgetting;
use App\Models\Department;
use App\Models\Notifications;
use App\Models\SparePart;
use App\Models\StockInOut;
use App\Models\Supplier;
use App\Models\User;
use Carbon\Carbon;
use DateTime;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Dompdf\Dompdf as PDF;
use Mpdf\Mpdf;
use PhpOffice\PhpSpreadsheet\Worksheet\PageSetup;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Mail;
use PHPUnit\Framework\Error\Notice;
use ZipArchive;

class InventoryController extends Controller
{
    public function showSparePart(Request $request)
    {
        if(Auth::check()){

            $dropFilter = SparePart::where('is_deleted', 0)->orderBy('sp_id', 'DESC');

            $seriesFilter = $dropFilter->where('sp_series', '!=', null)->pluck('sp_series')->unique();
            $itemnoFilter = $dropFilter->where('sp_itemno', '!=', null)->pluck('sp_itemno')->unique();
            $lineFilter = $dropFilter->where('sp_line', '!=', null)->pluck('sp_line')->unique();
            $locationFilter = $dropFilter->where('sp_location', '!=', null)->pluck('sp_location')->unique();
            $priceFilter = $dropFilter->where('sp_unitprice', '!=', null)->pluck('sp_unitprice')->unique();
            $stockFilter = $dropFilter->where('sp_opening_stock', '!=', null)->pluck('sp_opening_stock')->unique();
            $minstockFilter = $dropFilter->where('sp_min_stock', '!=', null)->pluck('sp_min_stock')->unique();
            $descriptionFilter = $dropFilter->where('sp_desc', '!=', null)->pluck('sp_desc')->unique();
            $dieFilter = $dropFilter->where('sp_die_name', '!=', null)->pluck('sp_die_name')->unique();

            $filterSupplier = $request->input('supplier');
            $filterCategory = $request->input('category');
            $filterSeries = $request->input('series');
            $filterDepartment = $request->input('department');
            $filterItemno = $request->input('itemno');
            $filterLine = $request->input('line');
            $filterLocation = $request->input('location');
            $filterUnitprice = $request->input('unit');
            $filterStock = $request->input('stock');
            $filterMinstock = $request->input('min');
            $filterDescription = $request->input('desc');
            $filterDiename = $request->input('die');
            $filterRank = $request->input('rank');
            $filterItemstatus = $request->input('status');

            $query = SparePart::join('supplier', 'supplier_id', '=', 'sp_supplier_id')
            ->join('department', 'department_id', '=', 'sp_department')
            ->where('spare_part.is_deleted', 0);

            // Apply filters
            if ($filterSupplier) {
                $query->where('supplier_name', $filterSupplier);
            }
            if ($filterCategory) {
                $query->where('sp_category', $filterCategory);
            }
            if ($filterSeries) {
                $query->where('sp_series','LIKE','%'.$filterSeries.'%');
            }
            if ($filterDepartment) {
                $query->where('department_name', $filterDepartment);
            }
            if ($filterItemno) {
                $query->where('sp_itemno','LIKE','%'.$filterItemno.'%');
            }
            if ($filterLine) {
                $query->where('sp_line','LIKE','%'.$filterLine.'%');
            }
            if ($filterLocation) {
                $query->where('sp_location','LIKE','%'.$filterLocation.'%');
            }
            if ($filterUnitprice) {
                $query->where('sp_unitprice','LIKE','%'.$filterUnitprice.'%');
            }
            if ($filterStock) {
                $query->where('sp_opening_stock','LIKE','%'.$filterStock.'%');
            }
            if ($filterMinstock) {
                $query->where('sp_min_stock','LIKE','%'.$filterMinstock.'%');
            }
            if ($filterDescription) {
                $query->where('sp_desc','LIKE','%'.$filterDescription.'%');
            }
            if ($filterDiename) {
                $query->where('sp_die_name','LIKE','%'.$filterDiename.'%');
            }
            if ($filterRank) {
                $query->where('sp_rank','LIKE','%'.$filterRank.'%');
            }
            if ($filterItemstatus) {
                $query->where('sp_status','LIKE','%'.$filterItemstatus.'%');
            }

            $spare_part = $query->orderBy('sp_id', 'DESC')->get();

            $department = Department::where('is_deleted', 0)->get();
            // $department = Department::join('budget', 'budget_department', 'department_id')->where('budget.is_deleted', 0)->get();
            // $spare_part = SparePart::join('supplier', 'supplier_id','=','sp_supplier_id')
            //                         // ->join('budget', 'budget_id','=','sp_department')
            //                         ->join('department', 'department_id','=','sp_department')
            //                         ->where('spare_part.is_deleted', 0)
            //                         ->orderBy('sp_id', 'DESC')
            //                         ->get();

            $supplier = Supplier::where('is_deleted', 0)
                        ->orderBy('supplier_name', 'ASC')
                        ->get();

            $notification = Notifications::orderBy('notification_is_read', 'ASC')->get();
            $new_notification = Notifications::where('notification_is_read', 0)->count();

            return view('spare-part')->with([
                'department' => $department,
                'spare_part' => $spare_part,
                'supplier' => $supplier,
                'notification' => $notification,
                'new_notification' => $new_notification,
                'filter_series' => $seriesFilter,
                'filter_itemno' => $itemnoFilter,
                'filter_line' => $lineFilter,
                'filter_location' =>$locationFilter,
                'filter_unit' => $priceFilter,
                'filter_stock' => $stockFilter,
                'filter_min' => $minstockFilter,
                'filter_desc' => $descriptionFilter,
                'filter_die' => $dieFilter
            ]);
        }
        return redirect()->to('login')->withSuccess('You are not allowed to access');

    }

    public function showStockInOut(Request $request)
    {
        if(Auth::check()){

            $dropFilter = StockInOut::join('spare_part', 'spare_part.sp_id','=','stock_inout.si_sp_id')
                    ->where('si_type', 'Stock In')
                    ->where('stock_inout.is_deleted', 0)
                    ->orderBy('si_id', 'DESC');

            $quantityFilter = $dropFilter->whereNotNull('si_quantity')->pluck('si_quantity')->unique();
            $descriptionFilter = $dropFilter->whereNotNull('sp_desc')->pluck('sp_desc')->unique();
            $seriesFilter = $dropFilter->pluck('sp_series')->unique();
            $itemlotnoFilter = $dropFilter->pluck('si_itemlotno');
            $itemnoFilter = $dropFilter->whereNotNull('sp_itemno')->pluck('sp_itemno');
            $dieFilter = $dropFilter->whereNotNull('sp_die_name')->pluck('sp_die_name')->unique();
            $dateFilter = $dropFilter->whereNotNull('stock_inout.created_at')->pluck('stock_inout.created_at')->unique();

            $filterSeries = $request->input('series');
            $filterDie = $request->input('die');
            $filterItemNo = $request->input('itemno');
            $filterItemLotNo = $request->input('itemlotno');
            $filterDescription = $request->input('desc');
            $filterSupplier = $request->input('supplier');
            $filterQuantity = $request->input('quantity');
            $filterUser = $request->input('user');
            $filterDate = $request->input('date');

            $query = StockInOut::select('stock_inout.si_id','spare_part.sp_id','spare_part.sp_series','spare_part.sp_die_name','stock_inout.si_itemlotno','spare_part.sp_desc','supplier.supplier_name','stock_inout.si_quantity','users.name','stock_inout.created_at', 'stock_inout.si_attachment','spare_part.sp_itemno','spare_part.sp_line','spare_part.sp_unitprice','stock_inout.si_unitprice')
                    ->join('spare_part', 'sp_id','=','si_sp_id')
                    ->join('supplier', 'supplier_id','sp_supplier_id')
                    ->join('users', 'id','=','si_who')
                    ->where('si_type', 'Stock In')
                    ->where('stock_inout.is_deleted', 0);

            if ($filterSeries) {
                $query->where('sp_series', 'LIKE', '%'.$filterSeries.'%');
            }
            if ($filterDie) {
                $query->where('sp_die_name', 'LIKE', '%'.$filterDie.'%');
            }
            if ($filterItemNo) {
                $query->where('sp_itemno', 'LIKE', '%'.$filterItemNo.'%');
            }
            if ($filterItemLotNo) {
                $query->where('si_itemlotno', 'LIKE', '%'.$filterItemLotNo.'%');
            }
            if ($filterDescription) {
                $query->where('sp_desc', 'LIKE', '%'.$filterDescription.'%');
            }
            if ($filterSupplier) {
                $query->where('supplier_name', $filterSupplier);
            }
            if ($filterQuantity) {
                $query->where('si_quantity', $filterQuantity);
            }
            if ($filterUser) {
                $query->where('name', $filterUser);
            }
            if ($filterDate) {
                $query->whereDate('stock_inout.created_at', $filterDate);
            }

            $stockIn = $query->orderBy('si_id', 'DESC')->get();

            // $in = StockInOut::select('stock_inout.si_id','spare_part.sp_id','spare_part.sp_series','spare_part.sp_die_name','stock_inout.si_itemlotno','spare_part.sp_desc','supplier.supplier_name','stock_inout.si_quantity','users.name','stock_inout.created_at', 'stock_inout.si_attachment','spare_part.sp_itemno')
            //         ->join('spare_part', 'sp_id','=','si_sp_id')
            //         ->join('supplier', 'supplier_id','sp_supplier_id')
            //         ->join('users', 'id','=','si_who')
            //         ->where('si_type', 'Stock In')
            //         ->where('stock_inout.is_deleted', 0)
            //         ->orderBy('si_id', 'DESC')
            //         ->get();

            $out = StockInOut::select('stock_inout.si_id','spare_part.sp_id','spare_part.sp_series','spare_part.sp_die_name','stock_inout.si_itemlotno','spare_part.sp_desc','supplier.supplier_name','stock_inout.si_quantity','users.name','stock_inout.created_at', 'stock_inout.si_attachment','spare_part.sp_itemno','spare_part.sp_line')
                    ->join('spare_part', 'sp_id','=','si_sp_id')
                    ->join('supplier', 'supplier_id','sp_supplier_id')
                    ->join('users', 'id','=','si_who')
                    ->where('si_type', 'Stock Out')
                    ->where('stock_inout.is_deleted', 0)
                    ->orderBy('si_id', 'DESC')
                    ->get();

            $user = User::where('is_active', 1)
                    ->orderBy('name', 'ASC')
                    ->get();

            $supplier = Supplier::where('is_deleted', 0)
                        ->orderBy('supplier_name', 'ASC')
                        ->get();

            $notification = Notifications::orderBy('notification_is_read', 'ASC')->get();
            $new_notification = Notifications::where('notification_is_read', 0)->count();

            return view('stockinout')->with([
                'in' => $stockIn,
                'out' => $out,
                'notification' => $notification,
                'new_notification' => $new_notification,
                'user' => $user,
                'supplier' => $supplier,
                'filter_series' => $seriesFilter,
                'filter_die' => $dieFilter,
                'filter_itemno' => $itemnoFilter,
                'filter_itemlotno' => $itemlotnoFilter,
                'filter_desc' => $descriptionFilter,
                'filter_quantity' => $quantityFilter,
                'filter_date' => $dateFilter
            ]);
        }
        return redirect()->to('login')->withSuccess('You are not allowed to access');

    }

    public function showStockOut(Request $request)
    {
        if(Auth::check()){

            $dropFilter = StockInOut::join('spare_part', 'spare_part.sp_id','=','stock_inout.si_sp_id')
                    ->where('si_type', 'Stock Out')
                    ->where('stock_inout.is_deleted', 0)
                    ->orderBy('si_id', 'DESC');

            $quantityFilter = $dropFilter->whereNotNull('si_quantity')->pluck('si_quantity')->unique();
            $descriptionFilter = $dropFilter->whereNotNull('sp_desc')->pluck('sp_desc')->unique();
            $seriesFilter = $dropFilter->pluck('sp_series')->unique();
            $itemlotnoFilter = $dropFilter->pluck('si_itemlotno');
            $itemnoFilter = $dropFilter->whereNotNull('sp_itemno')->pluck('sp_itemno');
            $dieFilter = $dropFilter->whereNotNull('sp_die_name')->pluck('sp_die_name')->unique();
            $dateFilter = $dropFilter->whereNotNull('stock_inout.created_at')->pluck('stock_inout.created_at')->unique();


            $filterSeries = $request->input('series');
            $filterDie = $request->input('die');
            $filterItemNo = $request->input('itemno');
            $filterItemLotNo = $request->input('itemlotno');
            $filterDescription = $request->input('desc');
            $filterSupplier = $request->input('supplier');
            $filterQuantity = $request->input('quantity');
            $filterUser = $request->input('user');
            $filterDate = $request->input('date');

            $query = StockInOut::select('stock_inout.si_id','spare_part.sp_id','spare_part.sp_series','spare_part.sp_die_name','stock_inout.si_itemlotno','spare_part.sp_desc','supplier.supplier_name','stock_inout.si_quantity','users.name','stock_inout.created_at', 'stock_inout.si_attachment','spare_part.sp_itemno','spare_part.sp_line','spare_part.sp_unitprice')
                    ->join('spare_part', 'sp_id','=','si_sp_id')
                    ->join('supplier', 'supplier_id','sp_supplier_id')
                    ->join('users', 'id','=','si_who')
                    ->where('si_type', 'Stock Out')
                    ->where('stock_inout.is_deleted', 0);

            if ($filterSeries) {
                $query->where('sp_series', 'LIKE', '%'.$filterSeries.'%');
            }
            if ($filterDie) {
                $query->where('sp_die_name', 'LIKE', '%'.$filterDie.'%');
            }
            if ($filterItemNo) {
                $query->where('sp_itemno', 'LIKE', '%'.$filterItemNo.'%');
            }
            if ($filterItemLotNo) {
                $query->where('sp_itemlotno', 'LIKE', '%'.$filterItemLotNo.'%');
            }
            if ($filterDescription) {
                $query->where('sp_desc', 'LIKE', '%'.$filterDescription.'%');
            }
            if ($filterSupplier) {
                $query->where('supplier_name', $filterSupplier);
            }
            if ($filterQuantity) {
                $query->where('si_quantity', $filterQuantity);
            }
            if ($filterUser) {
                $query->where('name', $filterUser);
            }
            if ($filterDate) {
                $query->whereDate('stock_inout.created_at', $filterDate);
            }

            $stockout = $query->orderBy('si_id', 'DESC')->get();

            // $out = StockInOut::select('stock_inout.si_id','spare_part.sp_id','spare_part.sp_series','spare_part.sp_die_name','stock_inout.si_itemlotno','spare_part.sp_desc','supplier.supplier_name','stock_inout.si_quantity','users.name','stock_inout.created_at', 'stock_inout.si_attachment','spare_part.sp_itemno')
            //         ->join('spare_part', 'sp_id','=','si_sp_id')
            //         ->join('supplier', 'supplier_id','sp_supplier_id')
            //         ->join('users', 'id','=','si_who')
            //         ->where('si_type', 'Stock Out')
            //         ->where('stock_inout.is_deleted', 0)
            //         ->orderBy('si_id', 'DESC')
            //         ->get();

            $user = User::where('is_active', 1)
                    ->orderBy('name', 'ASC')
                    ->get();

            $supplier = Supplier::where('is_deleted', 0)
                        ->orderBy('supplier_name', 'ASC')
                        ->get();

            $notification = Notifications::orderBy('notification_is_read', 'ASC')->get();
            $new_notification = Notifications::where('notification_is_read', 0)->count();

            return view('stockout')->with([
                'out' => $stockout,
                'notification' => $notification,
                'new_notification' => $new_notification,
                'user' => $user,
                'supplier' => $supplier,
                'filter_series' => $seriesFilter,
                'filter_die' => $dieFilter,
                'filter_itemno' => $itemnoFilter,
                'filter_itemlotno' => $itemlotnoFilter,
                'filter_desc' => $descriptionFilter,
                'filter_quantity' => $quantityFilter,
                'filter_date' => $dateFilter
            ]);
        }
        return redirect()->to('login')->withSuccess('You are not allowed to access');

    }

    public function showAddStockIn()
    {
        if(Auth::check()){

            $supplier = Supplier::where('is_deleted', 0)
                        ->orderBy('supplier_name', 'ASC')
                        ->get();
            $part = SparePart::join('supplier', 'supplier_id','=','sp_supplier_id')
                    ->where('spare_part.is_deleted', 0)
                    ->orderBy('sp_id', 'DESC')
                    ->get();
            $today = new DateTime(Carbon::now());
            $now = date_format($today, 'd/m/Y');

            $notification = Notifications::orderBy('notification_is_read', 'ASC')->get();
            $new_notification = Notifications::where('notification_is_read', 0)->count();

            return view('addstockin')->with([
                'supplier' => $supplier,
                'part' => $part,
                'now' => $now,
                'notification' => $notification,
                'new_notification' => $new_notification,
            ]);
        }
        return redirect()->to('login')->withSuccess('You are not allowed to access');

    }

    public function showAddStockOut()
    {
        if(Auth::check()){

            $supplier = Supplier::where('is_deleted', 0)
                        ->orderBy('supplier_name', 'ASC')
                        ->get();

            // $part = SparePart::join('supplier', 'supplier.supplier_id','=','spare_part.sp_supplier_id')
            //         ->join('stock_inout', 'stock_inout.si_sp_id','=','spare_part.sp_id')
            //         ->where('spare_part.is_deleted', 0)
            //         ->where('stock_inout.si_type', 'Stock In')
            //         ->orderBy('sp_id', 'DESC')
            //         ->get();

                    $part = SparePart::join('supplier', 'supplier_id','=','sp_supplier_id')
                    ->where('spare_part.is_deleted', 0)
                    ->orderBy('sp_id', 'DESC')
                    ->get();

                    // $part = SparePart::with(['supplier', 'stockInOut' => function($query) {
                    //     $query->where('is_deleted', 0)
                    //           ->where('si_type', 'Stock In');
                    // }])
                    // ->where('sp_id', 220)
                    // ->orderBy('sp_id', 'DESC')
                    // ->get();
                    // dd($part);

            // // Loop through the parts and their related stockInOut data
            // foreach($part as $parts) {
            //     $supplierName = $parts->supplier->supplier_name;
            //     foreach($parts->stockInOut as $stock) {

            //     }
            // }

            $today = new DateTime(Carbon::now());
            $now = date_format($today, 'd/m/Y');

            $notification = Notifications::orderBy('notification_is_read', 'ASC')->get();
            $new_notification = Notifications::where('notification_is_read', 0)->count();

            return view('addstockout')->with([
                'supplier' => $supplier,
                'part' => $part,
                'now' => $now,
                'notification' => $notification,
                'new_notification' => $new_notification,
            ]);
        }
        return redirect()->to('login')->withSuccess('You are not allowed to access');

    }

    public function addSparePart(Request $request)
    {
        $attach = $request['drawing'];
        $attachName = null;
        if($attach != null){
            $attachName = $attach->getClientOriginalName();
            $attachType = $attach->getClientOriginalExtension();
            $attachSize = $attach->getSize();
            $fileSizeInMB = $attachSize / (1024 * 1024);

            // Limit the File type
            if($attachType != 'jpg' &&
                $attachType != 'jpeg' &&
                $attachType != 'png' &&
                $attachType != 'pdf' &&
                $attachType != 'tif'){
                return back()->with('error3', 'Sorry, only JPG, JPEG, PNG, PDF & TIF files are allowed.');
            }

            //Limit the File Size
            if($fileSizeInMB > 200){
                return back()->with('error4', 'Sorry, the maximum allowable file size is limited to 200 MB.');
            }
        }

        $existingItem = SparePart::where('sp_itemno', $request['itemno'])
                        ->where('sp_supplier_id', $request['name'])
                        ->where('is_deleted', 0)
                        ->exists();
        $existingItemDeleted = SparePart::where('sp_itemno', $request['itemno'])
                            ->where('sp_supplier_id', $request['name'])
                            ->where('is_deleted', 1)
                            ->exists();

        if ($existingItem) {
            return redirect()->back()->with('error_solo1', "Item number ".$request['itemno']." already exists.");
        }
        if ($existingItemDeleted) {
            return redirect()->back()->with('error_solo2', "Item with item number ".$request['itemno']." already exists in restore module.");
        }

        $department_id = Auth::user()->user_department_id;

        $part = new SparePart();
        $part->sp_supplier_id = $request['name'];
        $part->sp_category = $request['category'];
        $part->sp_series = $request['series'];
        $part->sp_itemno = $request['itemno'];
        $part->sp_line = $request['line'];
        $part->sp_department = $department_id;
        $part->sp_location = $request['location'];
        $part->sp_unitprice = $request['unit'];
        $part->sp_opening_stock = $request['stock'];
        $part->sp_opening_amount = $request['amount'];
        $part->sp_min_stock = $request['min'];
        $part->sp_desc = $request['desc'];
        $part->sp_die_name = $request['die'];
        $part->sp_rank = $request['rank'];
        $part->sp_status = $request['status'];
        $part->sp_drawing = $attachName;
        $part->save();

        //Add in stock in for item lot number
        $stockin = new StockInOut();
        $stockin->si_sp_id = $part->sp_id;
        $stockin->si_type = 'Stock In';
        $stockin->si_quantity = $request['stock'];
        $stockin->si_unitprice = $request['unit'];
        $stockin->si_total = $request['stock'] * $request['unit'];
        $stockin->si_itemlotno = $request['itemlotno'];
        $stockin->si_who = Auth::user()->id;
        $stockin->save();

        if($attach != null){
            $attach->storeAs('spare_part/'.$part->sp_id, $attachName);
        }

        $audit = new Audit();
        $audit->at_module = 'Inventory Tracking';
        $audit->at_submodule = 'Spare Part';
        $audit->at_activity = 'Add Spare Part';
        $audit->at_description = 'Add Spare Part Item No: '.$part->sp_itemno
                                .'<br>Item description: '.$part->sp_desc
                                .'<br>Lot number:'.$request['itemlotno']
                                .'<br>Quantity: '.$part->sp_opening_stock.'pcs'
                                .'<br>Current Stock: '.$part->sp_opening_stock.'pcs';
        $audit->at_user_id = Auth::user()->id;
        $audit->save();

        return redirect()->back();
    }

    public function addSparePartBulk(Request $request)
    {

        $request->validate([
            'file' => 'required|mimes:xlsx',
        ]);

        $file = $request->file('file');
        // $supplier_name = $request->input('name');
        // $category = $request->input('category');
        // $department = $request->input('department');
        // $rank = $request->input('rank');
        // $status = $request->input('status');
        // $drawing = $request->input('drawing');

        $fileName = $file->getClientOriginalName();

        $data = Excel::toArray([], $file);

        // Skip the first header in excel
        $data = array_slice($data[0], 1);

        $errors = [];
        $consolidatedErrors = [];

        $no = 1;

        $audit = new Audit();
        $audit->at_module = 'Inventory Tracking';
        $audit->at_submodule = 'Spare Part';
        $audit->at_activity = 'Add Spare Part by Group';
        $audit->at_description = null;
        $audit->at_user_id = Auth::user()->id;
        $audit->save();

        foreach ($data as $row) {
            // if(!is_numeric($row[4]) || !is_numeric($row[5]) || !is_numeric($row[6])){
            //     return back()->with('error2', 'Kindly input numerical values in the Unit Price, Opening Stock, and Minimum Safety Stock columns. Please check it in excel file.');
            // }

            $supplier_id = Supplier::where('supplier_name', $row[0])->value('supplier_id');
            $department_id = Department::where('department_name', $row[2])->value('department_id');
            $existingItem = SparePart::where('sp_itemno', $row[4])
                            ->where('sp_supplier_id', $supplier_id)
                            ->exists();

            $errors = [];

            // Validate the existence of department_id
            if ($department_id === null) {
                $errors[] = "Department with name ".$row[2]." does not exist for item number ".$row[4].".";
            }

            // Check if the supplier_id is null
            if ($supplier_id === null) {
                $errors[] = "Supplier with name ".$row[0]." does not exist or misspelled for item number ".$row[4].".";
            }

             // Check if the item_no already exists
            if ($existingItem) {
                $errors[] = "Item with item number ".$row[4]." already exists.";
            }

            // If there are errors for this item, consolidate them into a single error message
            if (!empty($errors)) {
                $consolidatedErrors[] = " - " . implode(PHP_EOL . ' - ', $errors);
            }

            // If the supplier_id, department_id, and item_no do not exist, insert the data
            if ($supplier_id !== null && $department_id !== null && !$existingItem) {
                $part = SparePart::create([
                    'sp_supplier_id' => $supplier_id,
                    'sp_category' => $row[1],
                    'sp_department' => $department_id,
                    'sp_series' => $row[3],
                    'sp_itemno' => $row[4],
                    'sp_line' => $row[6],
                    'sp_location' => $row[7],
                    'sp_unitprice' => $row[8],
                    'sp_opening_stock' => $row[9],
                    'sp_opening_amount' => $row[8] * $row[9],
                    'sp_min_stock' => $row[10],
                    'sp_desc' => $row[11],
                    'sp_die_name' => $row[12],
                    'sp_rank' => $row[13],
                    'sp_status' => $row[14]
                    // 'sp_drawing' => $drawing
                ]);

                // Add to stock in for itemlotno
                $stockin = new StockInOut();
                $stockin->si_sp_id = $part->sp_id;
                $stockin->si_type = 'Stock In';
                $stockin->si_quantity = $row[9];
                $stockin->si_unitprice = $row[8];
                $stockin->si_total =  $row[8] * $row[9];
                $stockin->si_itemlotno = $row[5];
                $stockin->si_who = Auth::user()->id;
                $stockin->save();

                $auditDescription = 'Added Spare Parts by Group in excel ('.$no.')';
                $auditDescription .= '<br>Item number: ' . $row[4];
                $auditDescription .= '<br>Item description: ' . $row[11];
                $auditDescription .= '<br>Lot number: ' . $row[5];
                $auditDescription .= '<br>Quantity: ' . $row[9].'pcs';

                $audit_desc = new AuditDesc();
                $audit_desc->ad_audit_id = $audit->at_id;
                $audit_desc->ad_description = $auditDescription;
                $audit_desc->save();

                $no++;
            }
        }

        // Check if there are any consolidated errors
        if (!empty($consolidatedErrors)) {
            return redirect()->back()->with('errors', $consolidatedErrors);
        }

        return redirect()->back()->with('success', 'Data imported successfully!');

    }

    public function downloadAttachment($sp_id, $file)
    {
        $path = storage_path('app/attachment/'.$sp_id.'/' . $file);

        if (file_exists($path)) {
            return response()->download($path);
        } else {
            abort(404); // File not found
        }

    }

    public function downloadTemplateSparepart()
    {
        $supplier = Supplier::where('is_deleted', 0)->get();
        $supplier_names = $supplier->pluck('supplier_name')->toArray();
        $department = Department::where('is_deleted', 0)->get();
        $department_names = $department->pluck('department_name')->toArray();
        $category = ['SS', 'SK'];

        $spreadsheet = IOFactory::load('template/Add_Sparepart_Bulk.xlsx');
        $worksheet = $spreadsheet->getActiveSheet();

        // Set data validation for the entire column A starting from A2 and below
        $column = 'A';
        $range = $column . '2:' . $column . '1048576'; // Set a large number to cover all rows

        $validation = $worksheet->getDataValidation($range);
        $validation->setType(\PhpOffice\PhpSpreadsheet\Cell\DataValidation::TYPE_LIST);
        $validation->setErrorTitle('Input error');
        $validation->setError('Value is not in the list');
        $validation->setShowDropDown(true);
        $validation->setFormula1('"'.implode(',', $supplier_names).'"');

        // Set data validation for the entire column B starting from B2 and below
        $columnB = 'B';
        $rangeB = $columnB . '2:' . $columnB . '1048576'; // Set a large number to cover all rows

        $validationB = $worksheet->getDataValidation($rangeB);
        $validationB->setType(\PhpOffice\PhpSpreadsheet\Cell\DataValidation::TYPE_LIST);
        $validationB->setErrorTitle('Input error');
        $validationB->setError('Value is not in the list');
        $validationB->setShowDropDown(true);
        $validationB->setFormula1('"'.implode(',', $category).'"');

        // Set data validation for the entire column C starting from C2 and below
        $column = 'C';
        $range = $column . '2:' . $column . '1048576'; // Set a large number to cover all rows

        $validation = $worksheet->getDataValidation($range);
        $validation->setType(\PhpOffice\PhpSpreadsheet\Cell\DataValidation::TYPE_LIST);
        $validation->setErrorTitle('Input error');
        $validation->setError('Value is not in the list');
        $validation->setShowDropDown(true);
        $validation->setFormula1('"'.implode(',', $department_names).'"');

        // Save the updated Excel file
        $writer = new Xlsx($spreadsheet);
        $writer->save('template/Add_Sparepart_Bulk_With_DataValidation.xlsx');
        $file_path = public_path('template/Add_Sparepart_Bulk_With_DataValidation.xlsx');
        $headers = [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ];

        return response()->download($file_path, 'Add_Sparepart_Bulk_With_DataValidation.xlsx', $headers);


        // $file_path = public_path('template/Add_Sparepart_Bulk.xlsx');
        // $headers = [
        //     'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        // ];
        // return response()->download($file_path, 'Add_Sparepart_Bulk.xlsx', $headers);
    }

    public function showEditSparePart($sp_id)
    {
        if(Auth::check()){
            $part = SparePart::join('supplier', 'supplier_id','=','sp_supplier_id')
                            // ->join('budget', 'budget_id','=','sp_department')
                            ->join('department', 'department_id','=','sp_department')
                            ->where('spare_part.sp_id', $sp_id)
                            ->first();

            $supplier = Supplier::where('is_deleted', 0)
                        ->orderBy('supplier_name', 'ASC')
                        ->get();
            $department = Department::where('is_deleted', 0)
                        ->orderBy('department_name', 'ASC')
                        ->get();
            // $department = Department::join('budget', 'budget_department', 'department_id')->where('budget.is_deleted', 0)->get();
            $notification = Notifications::orderBy('notification_is_read', 'ASC')->get();
            $new_notification = Notifications::where('notification_is_read', 0)->count();


            return view('edit-spare-part')->with([
                'part' => $part,
                'supplier' => $supplier,
                'department' => $department,
                'notification' => $notification,
                'new_notification' => $new_notification,
            ]);
        }
        return redirect()->to('login')->withSuccess('You are not allowed to access');
    }

    public function showEditStockInOut($si_id)
    {
        if(Auth::check()){

            $stockin = StockInOut::join('spare_part', 'sp_id','=','si_sp_id')
                    ->join('supplier', 'supplier_id','=','sp_supplier_id')
                    ->join('users', 'id','=','si_who')
                    ->where('stock_inout.si_id', $si_id)
                    ->first();

            $notification = Notifications::orderBy('notification_is_read', 'ASC')->get();
            $new_notification = Notifications::where('notification_is_read', 0)->count();


            return view('edit-stockin')->with([
                'notification' => $notification,
                'stockin' => $stockin,
                'new_notification' => $new_notification,
            ]);
        }
        return redirect()->to('login')->withSuccess('You are not allowed to access');
    }

    public function updateStockInOutForm(Request $request, $si_id)
    {
        $quantity = $request['quantity'];
        $itemlotno = $request['itemlotno'];
        $attachment = $request->file('attachment');
        $line = $request['line'];
        $reason = $request['reason'];

        $stockin = StockInOut::find($si_id);
        $stockin->si_itemlotno = $itemlotno;
        $stockin->si_line = $line;
        $stockin->si_reason = $reason;

        //Audit Trail
        $changes = [];

        $originalQuantity = $stockin->si_quantity;
        $originalItem = $stockin->si_itemlotno;
        $originalLine = $stockin->si_line;
        $originalReason = $stockin->si_reason;

        // Compare and record changes
        if ($originalQuantity != $quantity) {
            $changes[] = "<br>Quantity changed from '$originalQuantity' to '$quantity'";
        }
        if ($originalItem != $itemlotno) {
            $changes[] = "<br>Item Lot No changed from '$originalItem' to '$itemlotno'";
        }
        if($originalLine != $line){
            $changes[] = "<br>Line changed from '$originalLine' to '$line'";
        }
        if($originalReason != $reason){
            $changes[] = "<br>Cause/Reason changed from '$originalReason' to '$reason'";
        }


        // Create the audit description by joining the changes
        $itemno = SparePart::where('sp_id', $stockin->si_sp_id)->value('sp_itemno');
        $desc = SparePart::where('sp_id', $stockin->si_sp_id)->value('sp_desc');
        $auditDescription = 'Edited Stock In/Out: <br>
                            Item number: '.$itemno.'<br>
                            Description: '.$desc.'
                            ' . implode(', ', $changes);

        if($changes != null){
            $audit = new Audit();
            $audit->at_module = 'Inventory Tracking';
            $audit->at_submodule = 'Stock In/Out';
            $audit->at_activity = 'Edit Stock In/Out';
            $audit->at_description = $auditDescription;
            $audit->at_user_id = Auth::user()->id;
            $audit->save();
        }

        //If quantity was edited
        if($quantity != $stockin->si_quantity){
            $stockin->si_quantity = $quantity;

            $part = SparePart::find($stockin->si_sp_id);
            if($stockin->si_type == 'Stock Out'){
                // $part->sp_opening_stock = ($part->sp_opening_stock + $stockin->si_quantity) - $quantity;
                $part->sp_opening_stock = $part->sp_opening_stock - $quantity;
                // $part->sp_opening_amount = $part->sp_opening_stock * $part->sp_unitprice;
                // $part->save();
            }elseif($stockin->si_type == 'Stock In'){
                // $part = SparePart::find($stockin->si_sp_id);
                // $part->sp_opening_stock = ($part->sp_opening_stock - $stockin->si_quantity) + $quantity;
                $part->sp_opening_stock = $part->sp_opening_stock + $quantity;
            }
            $part->sp_opening_amount = $part->sp_opening_stock * $part->sp_unitprice;
            $part->save();
        }

        // If attachment was edited
        if($attachment != null){
            $attachName = $attachment->getClientOriginalName();
            $attachType = $attachment->getClientOriginalExtension();
            $attachSize = $attachment->getSize();
            $fileSizeInMB = $attachSize / (1024 * 1024); // Convert to MB

            // Limit the File type
            if($attachType != 'jpg' &&
                $attachType != 'jpeg' &&
                $attachType != 'png' &&
                $attachType != 'pdf' &&
                $attachType != 'tif'){
                return back()->with('error', 'Sorry, only JPG, JPEG, PNG, PDF & TIF files are allowed.');
            }

            //Limit the File Size
            if($fileSizeInMB > 200){
                return back()->with('error2', 'Sorry, the maximum allowable file size is limited to 200 MB.');
            }

            $attachment->storeAs('attachment/'.$si_id, $attachName);
            $stockin->si_attachment = $attachName;
        }

        $stockin->save();

        return redirect()->route('stockinout');
    }

    public function updateSparePart(Request $request, $sp_id)
    {
        $part = SparePart::find($sp_id);
        $attachment = $request->file('drawing');

        $originalSupplier = Supplier::where('supplier_id', $part->sp_supplier_id)->value('supplier_name');
        $newSupplier = Supplier::where('supplier_id', $request['name'])->value('supplier_name');
        $originalCategory = $part->sp_category;
        $originalSeries = $part->sp_series;
        $originalItemNo = $part->sp_itemno;
        $originalLine = $part->sp_line;
        $originalDepartment = Department::where('department_id', $part->sp_department)->value('department_name');
        $newDepartment = Department::where('department_id', $request['department'])->value('department_name');
        $originalLocation = $part->sp_location;
        $originalUnitPrice = $part->sp_unitprice;
        $originalStock = $part->sp_opening_stock;
        $originalMinStock = $part->sp_min_stock;
        $originalDesc = $part->sp_desc;
        $originalDieName = $part->sp_die_name;
        $originalRank = $part->sp_rank;
        $originalStatus = $part->sp_status;
        $originalDrawing = $part->sp_drawing;

        $part->sp_supplier_id = $request['name'];
        $part->sp_category = $request['category'];
        $part->sp_series = $request['series'];
        $part->sp_itemno = $request['itemno'];
        $part->sp_line = $request['line'];
        $part->sp_department = $request['department'];
        $part->sp_location = $request['location'];
        $part->sp_unitprice = $request['unit'];
        $part->sp_opening_stock = $request['stock'];
        $part->sp_opening_amount = $request['amount'];
        $part->sp_min_stock = $request['min'];
        $part->sp_desc = $request['desc'];
        $part->sp_die_name = $request['die'];
        $part->sp_rank = $request['rank'];
        $part->sp_status = $request['status'];

        // If attachment was edited
        if($attachment != null || $attachment != $originalDrawing){
            $attachName = $attachment->getClientOriginalName();
            $attachType = $attachment->getClientOriginalExtension();
            $attachSize = $attachment->getSize();
            $fileSizeInMB = $attachSize / (1024 * 1024); // Convert to MB

            // Limit the File type
            if($attachType != 'jpg' &&
                $attachType != 'jpeg' &&
                $attachType != 'png' &&
                $attachType != 'pdf' &&
                $attachType != 'tif'){
                return back()->with('error', 'Sorry, only JPG, JPEG, PNG, PDF & TIF files are allowed.');
            }

            //Limit the File Size
            if($fileSizeInMB > 200){
                return back()->with('error2', 'Sorry, the maximum allowable file size is limited to 200 MB.');
            }

            $attachment->storeAs('spare_part/'.$sp_id, $attachName);
            $part->sp_drawing = $attachName;
        }

        $changes = [];

        // Compare and record changes
        if ($originalSupplier !== $newSupplier) {
            $changes[] = "<br>Supplier changed from '$originalSupplier' to '$newSupplier'";
        }
        if ($originalCategory !== $request['category']) {
            $changes[] = "<br>Category changed from '$originalCategory' to '$request[category]'";
        }
        if ($originalSeries !== $request['series']) {
            $changes[] = "<br>Series changed from '$originalSeries' to '$request[series]'";
        }
        if ($originalItemNo !== $request['itemno']) {
            $changes[] = "<br>Item No changed from '$originalItemNo' to '$request[itemno]'";
        }
        if ($originalLine !== $request['line']) {
            $changes[] = "<br>Line changed from '$originalLine' to '$request[line]'";
        }
        if ($originalDepartment !== $newDepartment) {
            $changes[] = "<br>Department changed from '$originalDepartment' to '$newDepartment'";
        }
        if ($originalLocation !== $request['location']) {
            $changes[] = "<br>Location changed from '$originalLocation' to '$request[location]'";
        }
        if ($originalUnitPrice !== $request['unit']) {
            $changes[] = "<br>Unit Price changed from '$originalUnitPrice' to '$request[unit]'";
        }
        if ($originalStock != $request['stock']) {
            $changes[] = "<br>Opening Stock changed from '$originalStock' to '$request[stock]'";
        }
        if ($originalMinStock != $request['min']) {
            $changes[] = "<br>Minimum Stock changed from '$originalMinStock' to '$request[min]'";
        }
        if ($originalDesc !== $request['desc']) {
            $changes[] = "<br>Description changed from '$originalDesc' to '$request[desc]'";
        }
        if ($originalDieName !== $request['die']) {
            $changes[] = "<br>Die Name changed from '$originalDieName' to '$request[die]'";
        }
        if ($originalRank !== $request['rank']) {
            $changes[] = "<br>Rank changed from '$originalRank' to '$request[rank]'";
        }
        if ($originalStatus != $request['status']) {
            if($originalStatus == 1){ $originalStat = 'Active';}
            if($originalStatus == 2){ $originalStat = 'Slow Moving';}
            if($originalStatus == 0){ $originalStat = 'Non Active';}
            if($request['status'] == 1){ $reqStat = 'Active';}
            if($request['status'] == 2){ $reqStat = 'Slow Moving';}
            if($request['status'] == 0){ $reqStat = 'Non Active';}
            $changes[] = "<br>Status changed from '$originalStat' to '$reqStat'";
        }
        if ($originalDrawing !== $request['drawing']) {
            $changes[] = "<br>Drawing changed from '$originalDrawing' to '$request[drawing]'";
        }

        // Create the audit description by joining the changes
        $auditDescription = 'Edited Spare Part: <br>
                            Item number: '.$part->sp_itemno.'<br>
                            Description: '.$part->sp_desc.'
                            ' . implode(', ', $changes);

        if($changes != null){
            $audit = new Audit();
            $audit->at_module = 'Inventory Tracking';
            $audit->at_submodule = 'Spare Part';
            $audit->at_activity = 'Edit Spare Part';
            $audit->at_description = $auditDescription;
            $audit->at_user_id = Auth::user()->id;
            $audit->save();
        }

        $part->save();

        return redirect()->route('spare-part');
    }

    public function deleteSparePart($sp_id)
    {
        $part = SparePart::find($sp_id);
        $part->is_deleted = 1;
        $part->save();


        $stock = StockInOut::where('si_sp_id', $part->sp_id)->get();
        if($stock->count() != 0){
            foreach($stock as $stocks){
                $stocks->is_deleted = 1;
                $stocks->save();
            }
        }

        $audit = new Audit();
        $audit->at_module = 'Inventory Tracking';
        $audit->at_submodule = 'Spare Part';
        $audit->at_activity = 'Delete Spare Part';
        $audit->at_description = 'Deleted Spare Part <br>
                                 Item number: '.$part->sp_itemno.'<br>
                                 Description: '.$part->sp_desc;
        $audit->at_user_id = Auth::user()->id;
        $audit->save();

        return redirect()->back();

    }

    public function updateStock(Request $request, $si_id)
    {
        $stock = StockInOut::find($si_id);
        $stock->si_itemlotno = $request['itemlotno'];
        // $stock->si_quantity = $request['quantity'];
        $stock->save();

        $audit = new Audit();
        $audit->at_module = 'Inventory Tracking';
        $audit->at_submodule = 'Stock In/Out';
        $audit->at_activity = 'Edit Stock';
        $audit->at_description = 'Edit Stock:'. $si_id;
        $audit->at_user_id = Auth::user()->id;
        $audit->save();


        $spare = SparePart::where('sp_id', $stock->si_sp_id)->first();

        return redirect()->back();
    }

    public function deleteStock($si_id)
    {
        $stock = StockInOut::find($si_id);
        $stock->is_deleted = 1;
        $stock->save();

        $part = SparePart::where('sp_id', $stock->si_sp_id)->first();
        $part->sp_opening_stock -= $stock->quantity;
        $part->sp_opening_amount = $part->sp_opening_stock * $part->sp_unitprice;
        $part->save();

        $audit = new Audit();
        $audit->at_module = 'Inventory Tracking';
        $audit->at_submodule = 'Stock In/Out';
        $audit->at_activity = 'Delete Stock';
        $audit->at_description = 'Deleted Stock <br>
                                 Item number: '.$part->sp_itemno.'<br>
                                 Type: '. $stock->si_type.'<br>
                                 Lot no: '.$stock->si_itemlotno;
        $audit->at_user_id = Auth::user()->id;
        $audit->save();

        return redirect()->back();
    }

    public function updateStockIn(Request $request)
    {
        $id = $request->input('id');
        $stockInQty = $request->input('stockInQty');
        $itemLotNo = $request->input('itemLotNo');
        $attachment = $request->file('attachment');
        $unitPrice = $request->input('unitPrice');

        foreach($id as $index => $ids){
            $qty = $stockInQty[$index];
            $lotNo = isset($itemLotNo[$index]) ? $itemLotNo[$index] : '';
            $price = $unitPrice[$index];
            $sp_id = $id[$index];
            $attachName = null;
            if($attachment != null){
                $attach = $attachment[$index];
                $attachName = $attach->getClientOriginalName();
                $attachType = $attach->getClientOriginalExtension();
                $attachSize = $attach->getSize();
                $fileSizeInMB = $attachSize / (1024 * 1024); // Convert to MB

                // Limit the File type
                if($attachType != 'jpg' &&
                   $attachType != 'jpeg' &&
                   $attachType != 'png' &&
                   $attachType != 'pdf' &&
                   $attachType != 'tif'){
                    return back()->with('error', 'Sorry, only JPG, JPEG, PNG, PDF & TIF files are allowed.');
                }

                //Limit the File Size
                if($fileSizeInMB > 200){
                    return back()->with('error2', 'Sorry, the maximum allowable file size is limited to 200 MB.');
                }
            }

            //Add to stock_inout db
            $stockIn = new StockInOut();
            $stockIn->si_sp_id = $sp_id;
            $stockIn->si_type = 'Stock In';
            $stockIn->si_quantity = $qty;

            //If changes the price
            $part = SparePart::find($sp_id);
            if($price != $part->sp_unitprice){
                $new_part = $part->replicate();
                $new_part->sp_unitprice = $price;
                $new_part->sp_opening_stock = $qty;
                $new_part->sp_opening_amount = $price * $qty;
                $new_part->is_duplicate = 1;
                $new_part->created_at = $part->created_at;
                $new_part->updated_at = $part->updated_at;
                $new_part->save();

                $stockIn->si_unitprice = $price;

            }elseif($price == $part->sp_unitprice){
                $part->sp_opening_stock += $qty;
                // $part->sp_unitprice = $price;
                $part->sp_opening_amount = $part->sp_opening_stock * $price;
                $part->save();

                $stockIn->si_unitprice = $price;
            }

            // $stockIn = new StockInOut();
            // $stockIn->si_sp_id = $sp_id;
            // $stockIn->si_type = 'Stock In';
            // $stockIn->si_quantity = $qty;
            $stockIn->si_total = $price * $qty;
            $stockIn->si_itemlotno = $lotNo;
            $stockIn->si_attachment = $attachName;
            $stockIn->si_who = Auth::user()->id;
            $stockIn->save();

            if($attachment != null){
                $attach->storeAs('attachment/'.$sp_id, $attachName);
            }


            $audit = new Audit();
            $audit->at_module = 'Inventory Tracking';
            $audit->at_submodule = 'Stock In';
            $audit->at_activity = 'Add Stock In';
            $audit->at_description = 'Add Stock In: '.$part->sp_series
                                    .'<br>Item number: '.$part->sp_itemno
                                    .'<br>Item description: '.$part->sp_desc
                                    .'<br>Lot number: '.$lotNo
                                    .'<br>Quantity: '.$qty.'pcs'
                                    .'<br>Current Stock: '.$part->sp_opening_stock.'pcs';

            $audit->at_user_id = Auth::user()->id;
            $audit->save();

            $notification = Notifications::where('notification_sp_id', $sp_id)->get();
            if($notification != null){
                $notification->each->delete();
            }

        }
        return redirect()->route('stockinout');
    }

    public function updateStockOut(Request $request)
    {
        $id = $request->input('id');
        $stockOutQty = $request->input('stockOutQty');
        $line = $request->input('line');
        $reason = $request->input('reason');
        // $itemLotNo = $request->input('itemLotNo');
        // $attachment = $request->file('attachment');

        foreach($id as $index => $ids){
            $qty = $stockOutQty[$index];
            $sp_id = $id[$index];
            $lines = $line[$index];
            $reasons = $reason[$index];

            $part = SparePart::find($sp_id);
            if($qty > $part->sp_opening_stock){
                return back()->with('error', 'The requested quantity exceeds the available stock.');
            }else{
                $part->sp_opening_stock -= $qty;
                $part->sp_opening_amount = $part->sp_unitprice * $part->sp_opening_stock;
                $part->save();
            }

            $stockOut = new StockInOut();
            $stockOut->si_sp_id = $sp_id;
            $stockOut->si_type = 'Stock Out';
            $stockOut->si_quantity = $qty;
            $stockOut->si_line = $lines;
            $stockOut->si_reason = $reasons;
            $stockOut->si_who = Auth::user()->id;
            $stockOut->save();

            $stock_detail = StockInOut::where('si_sp_id', $sp_id)->first();

            $audit = new Audit();
            $audit->at_module = 'Inventory Tracking';
            $audit->at_submodule = 'Stock Out';
            $audit->at_activity = 'Add Stock Out';
            $audit->at_description = 'Add Stock Out: '.$stock_detail->si_itemlotno
                                    .'<br>Item number: '.$part->sp_itemno
                                    .'<br>Line: '.$lines
                                    .'<br>Reason: '.$reasons
                                    .'<br> Quantity: '.$qty.'pcs'
                                    .'<br>Current Stock: '.$part->sp_opening_stock.'pcs';

            $audit->at_user_id = Auth::user()->id;
            $audit->save();

            //Notification
            if($part->sp_opening_stock <= $part->sp_min_stock){
                $now = Carbon::now();
                $notification = new Notifications();
                $notification->notification_user_id = 0;
                $notification->notification_title = 'Low Stock !';
                $notification->notification_subtitle = 'Item number: '.$part->sp_itemno;
                $notification->notification_message = 'Low stock for inventory: <br>
                                                      Item number: '. $part->sp_itemno.'<br>
                                                      Description: '. $part->sp_desc.'<br>
                                                      Category: '.$part->sp_category.'<br>
                                                      Line: '.$part->sp_line.'<br><br>
                                                      Please restock as soon as possible.';
                $notification->notification_is_read = 0;
                $notification->notification_datetime = $now;
                $notification->notification_sp_id = $part->sp_id;
                $notification->save();

                $to = User::join('user_role', 'user_role.ur_id','=','users.user_role_id')
                    ->where('users.user_department_id', $part->sp_department)
                    ->where('user_role.ur_name', 'Admin Department')
                    // ->value('users.email');
                    ->get();

                if($to){
                    foreach($to as $data){
                        Mail::to($data->email)->send(new LowStock($part->sp_series, $part->sp_itemno, $part->sp_category));
                    }
                }


            }

        }
        return redirect()->route('stockout');
    }

    public function restoreSparePart($sp_id)
    {
        $part = SparePart::find($sp_id);
        $part->is_deleted = 0;
        $part->save();

        $audit = new Audit();
        $audit->at_module = 'Restore';
        $audit->at_submodule = 'Restore';
        $audit->at_activity = 'Restore Spare Part';
        $audit->at_description = 'Restore Spare Part <br>
                                  Item number: '.$part->sp_itemno.'<br>
                                  Description: '.$part->sp_desc;
        $audit->at_user_id = Auth::user()->id;
        $audit->save();

        return redirect()->route('restore');
    }

    public function permanentDeleteSparePart($sp_id)
    {
        $part = SparePart::find($sp_id);

        $audit = new Audit();
        $audit->at_module = 'Restore';
        $audit->at_submodule = 'Restore';
        $audit->at_activity = 'Delete Spare Part';
        $audit->at_description = 'Permanently Delete Spare Part: '.$part->sp_itemno;
        $audit->at_user_id = Auth::user()->id;
        $audit->save();

        $part->delete();

        return redirect()->back();
    }

    public function restoreStock($si_id)
    {
        $stock = StockInOut::find($si_id);
        $stock->is_deleted = 0;
        $stock->save();

        $part = SparePart::where('sp_id', $stock->si_sp_id)->first();

        $audit = new Audit();
        $audit->at_module = 'Restore';
        $audit->at_submodule = 'Restore';
        $audit->at_activity = 'Restore Stock In/Out';
        $audit->at_description = 'Restore Stock In/Out <br>
                                  Item number: '.$part->sp_itemno.'<br>
                                  Type: '.$stock->si_type.'<br>
                                  Lot no: '.$stock->si_itemlotno;
        $audit->at_user_id = Auth::user()->id;
        $audit->save();

        return redirect()->route('restore');
    }

    public function permanentDeleteStock($si_id)
    {
        $stock = StockInOut::find($si_id);

        $audit = new Audit();
        $audit->at_module = 'Restore';
        $audit->at_submodule = 'Restore';
        $audit->at_activity = 'Delete Stock In/Out';
        $audit->at_description = 'Permanently Delete Spare Part: '.$stock->si_id;
        $audit->at_user_id = Auth::user()->id;
        $audit->save();

        $stock->delete();

        return redirect()->back();
    }

    public function downloadSingleExcel($sp_id, $type)
    {

        $spareparts = SparePart::join('supplier', 'supplier_id','=','sp_supplier_id')
                    ->where('sp_id', $sp_id)
                    ->where('spare_part.is_deleted', 0)
                    ->first();

        $budget = Budgetting::where('budget_id', $spareparts->sp_department)->value('budget_budget');
        $budget_format = number_format($budget, 2);

        //Get Excel Template
        $spreadsheet = IOFactory::load(public_path('template/SparePartReport.xlsx'));

        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setCellValue('T3', 'RM '.$budget_format);
        $column = 10;
        $no = 1;
        $sum = 0;

        $stockInQty = StockInOut::where('si_sp_id', $sp_id)->where('si_type', 'Stock In')->count();
        $stockOutQty = StockInOut::where('si_sp_id', $sp_id)->where('si_type', 'Stock Out')->count();
        $closingQty = $spareparts->sp_opening_stock + $stockInQty - $stockOutQty;
        $latestStockin = StockInOut::where('si_sp_id', $sp_id)->where('si_type', 'Stock In')->orderBy('created_at', 'DESC')->value('created_at');
        $latestStockOut = StockInOut::where('si_sp_id', $sp_id)->where('si_type', 'Stock Out')->orderBy('created_at', 'DESC')->value('created_at');
        $rank = $spareparts->sp_rank;

        $now = Carbon::now();
        $now = new DateTime();
        $latestStockinDate = new DateTime($latestStockin);
        $interval = $latestStockinDate->diff($now);
        $month = round($interval->days / 30.44);

        //Month Rank
        // if($month == 0.0){$rank = '-';}
        // elseif($month <= 6){$rank = 'A';}
        // elseif($month >= 7 && $month < 12){$rank = 'B';}
        // elseif($month >= 13 && $month < 24){$rank = 'C';}
        // elseif($month >= 25 && $month < 48){$rank = 'D';}
        // elseif($month >= 49){$rank = 'E';}

        //Status
        if($spareparts->sp_status == 0){$status = 'Non Active';}
        elseif($spareparts->sp_status == 1){$status = 'Active';}
        elseif($spareparts->sp_status == 2){$status = 'Slow moving';}

        $data = [
            'sp_id' => $spareparts->sp_id,
            'series' => $spareparts->sp_series,
            'die_name' => $spareparts->sp_die_name,
            'itemno' => $spareparts->sp_itemno,
            'description' => $spareparts->sp_desc,
            'supplier' => $spareparts->supplier_name,
            'unit_price' => $spareparts->sp_unitprice,
            'opening_stock' => $spareparts->sp_opening_stock,
            'opening_amount' => $spareparts->sp_opening_amount,
            'stockin_qty' => $stockInQty,
            'stockin_amount' => number_format($stockInQty * $spareparts->sp_unitprice, 2),
            'stockout_qty' => $stockOutQty,
            'stockout_amount' => number_format($stockOutQty * $spareparts->sp_unitprice, 2),
            'closing_qty' => $closingQty,
            'closing_amount' => number_format($closingQty * $spareparts->sp_unitprice),
            'status' => $status,
            'latest_stockin_date' => $latestStockin != null ? date_format($latestStockin, 'd/m/y') : '-',
            'latest_stockout_date' => $latestStockOut != null ? date_format($latestStockOut, 'd/m/y') : '-',
            'month' => $month,
            'rank' => $rank
        ];

        $total = $closingQty * $spareparts->sp_unitprice;

        //Put data in each row excel
        $sheet->setCellValue('A'.$column, $no);
        $sheet->setCellValue('B'.$column, $data['series']);
        $sheet->setCellValue('C'.$column, $data['die_name']);
        $sheet->setCellValue('D'.$column, $data['itemno']);
        $sheet->setCellValue('E'.$column, $data['description']);
        $sheet->setCellValue('F'.$column, $data['supplier']);
        $sheet->setCellValue('G'.$column, 'RM '.$data['unit_price']);
        $sheet->setCellValue('H'.$column, $data['opening_stock']);
        $sheet->setCellValue('I'.$column, 'RM '.$data['opening_amount']);
        $sheet->setCellValue('J'.$column, $data['stockin_qty']);
        $sheet->setCellValue('K'.$column, 'RM '.$data['stockin_amount']);
        $sheet->setCellValue('L'.$column, $data['stockout_qty']);
        $sheet->setCellValue('M'.$column, 'RM '.$data['stockout_amount']);
        $sheet->setCellValue('N'.$column, $data['closing_qty']);
        $sheet->setCellValue('O'.$column, 'RM '.$data['closing_amount']);
        $sheet->setCellValue('P'.$column, $data['status']);
        $sheet->setCellValue('Q'.$column, $data['latest_stockin_date']);
        $sheet->setCellValue('R'.$column, $data['latest_stockout_date']);
        $sheet->setCellValue('S'.$column, $data['month']);
        $sheet->setCellValue('T'.$column, $data['rank']);

        //Centered all column in excel
        $centeredColumns = ['A','B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T'];
        foreach ($centeredColumns as $columns) {
            $sheet->getStyle($columns)->getAlignment()->setHorizontal(\PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER);
        }

        //Colored column in excel
        $sheet->getStyle('G'.$column)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('FFCCFF');
        $sheet->getStyle('H'.$column)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('FFFF99');
        $sheet->getStyle('J'.$column)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('FFFF99');
        $sheet->getStyle('L'.$column)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('FFFF99');
        $sheet->getStyle('N'.$column)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('CCFFCC');
        if($status == 'Non Active'){
            $sheet->getStyle('P'.$column)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('FF0000');
        }elseif($status == 'Active'){
            $sheet->getStyle('P'.$column)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('00B050');
        }elseif($status == 'Slow moving'){
            $sheet->getStyle('P'.$column)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('FFFF00');
        }

        //Set currency format
        $currencyFormat = 'RM #,##0.00';
        $sheet->getStyle('G'.$column)->getNumberFormat()->setFormatCode($currencyFormat);
        $sheet->getStyle('I'.$column)->getNumberFormat()->setFormatCode($currencyFormat);
        $sheet->getStyle('K'.$column)->getNumberFormat()->setFormatCode($currencyFormat);
        $sheet->getStyle('M'.$column)->getNumberFormat()->setFormatCode($currencyFormat);
        $sheet->getStyle('O'.$column)->getNumberFormat()->setFormatCode($currencyFormat);

        $no++;
        $column++;
        $sum += $total;

        //Total Expenses
        $sheet->setCellValue('T4', 'RM '.number_format($sum, 2));

        //Border in excel
        $borderStyle = \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN;
        $sheet->getStyle('A10:T'.($column - 1))->getBorders()->getAllBorders()->setBorderStyle($borderStyle);

        //Save and download
        $path = 'report/SparePartMulti_report.xlsx';
        $filePath = public_path($path);

        $filenameExcel = 'Spare_Part_'.$sp_id.'.xlsx';

        $writer = new Xlsx($spreadsheet);
        $writer->save(public_path($path));

        if($type == 'pdf'){

            $getting_file = public_path($path);
            $reader = IOFactory::createReader('Xlsx');
            $spreadsheet2 = $reader->load($getting_file);
            $spreadsheet2->getActiveSheet()->getPageSetup()->setOrientation(PageSetup::ORIENTATION_LANDSCAPE);
            $writers = IOFactory::createWriter($spreadsheet2, 'Mpdf');
            $pdf_file = 'Spare_Part_'.$sp_id.'.pdf';

            $writers->save($pdf_file);

            return response()->download(public_path('Spare_Part_'.$sp_id.'.pdf'), 'Spare_Part_'.$sp_id.'.pdf');

        }else{
            return response()->download($filePath, $filenameExcel);
        }

    }

    public function downloadMultiExcel(Request $request)
    {
        set_time_limit(300);

        $department = $request['department'];
        $date = $request['month'];
        $type = $request['type'];

        // $sp_ids = $request->input('selectedValues');
        // $ids = explode(',', $sp_ids);
        // if($sp_ids == null){
        //     return redirect()->back()->with('excel','Please choose data to generate first');
        // }

        $filePaths = [];

        foreach($department as $departments){
            $spareparts = SparePart::select('spare_part.*','supplier.supplier_id', 'supplier.supplier_name')
                        ->join('supplier', 'supplier_id','=','sp_supplier_id')
                        // ->join('budget', 'budget_id','=','sp_department')
                        // ->whereIn('sp_id', $ids)
                        ->where('sp_department', $departments)
                        ->where('spare_part.is_deleted', 0)
                        // ->where('budget_month', $date)
                        // ->groupBy('sp_department')
                        ->get();

            //Get Budget
            $budget = Budgetting::where('budget_department', $departments)
                    ->where('budget_month', $date)
                    ->value('budget_budget');
            if($budget == null){
                return back()->with('month', 'Budget Month does not exist');
            }
            $budget_format = number_format($budget, 2);

            //Total Expenses in Stock In
            $year = date('Y', strtotime($date));
            $month = date('m', strtotime($date));
            $month_year = date('m/Y', strtotime($date));
            $total_expenses = StockInOut::whereRaw("YEAR(created_at) = ? AND MONTH(created_at) = ?", [$year, $month])->sum('si_total');
            $department_name = Department::where('department_id', $departments)->value('department_name');

            //Get Excel Template
            $spreadsheet = IOFactory::load(public_path('template/SparePartReport.xlsx'));

            $sheet = $spreadsheet->getActiveSheet();
            $sheet->setCellValue('T3', 'RM '.$budget_format);
            $sheet->setCellValue('T5', $department_name);
            $sheet->setCellValue('T6', $month_year);

            $column = 10;
            $no = 1;
            // $sum = 0;

            foreach($spareparts as $sparepart){

                $stockInQty = StockInOut::where('si_sp_id', $sparepart->sp_id)->where('si_type', 'Stock In')->count();
                $stockOutQty = StockInOut::where('si_sp_id', $sparepart->sp_id)->where('si_type', 'Stock Out')->count();
                $latestStockin = StockInOut::where('si_sp_id', $sparepart->sp_id)->where('si_type', 'Stock In')->orderBy('created_at', 'DESC')->value('created_at');
                $latestStockOut = StockInOut::where('si_sp_id', $sparepart->sp_id)->where('si_type', 'Stock Out')->orderBy('created_at', 'DESC')->value('created_at');

                $closingQty = $sparepart->sp_opening_stock + $stockInQty - $stockOutQty;
                $rank = $sparepart->sp_rank;

                $now = Carbon::now();
                $now = new DateTime();
                $latestStockinDate = new DateTime($latestStockin);
                $interval = $latestStockinDate->diff($now);
                $month = round($interval->days / 30.44);

                //Month Rank
                // if($month == 0.0){$rank = '-';}
                // elseif($month <= 6){$rank = 'A';}
                // elseif($month >= 7 && $month < 12){$rank = 'B';}
                // elseif($month >= 13 && $month < 24){$rank = 'C';}
                // elseif($month >= 25 && $month < 48){$rank = 'D';}
                // elseif($month >= 49){$rank = 'E';}

                //Status
                if($sparepart->sp_status == 0){$status = 'Non Active';}
                elseif($sparepart->sp_status == 1){$status = 'Active';}
                elseif($sparepart->sp_status == 2){$status = 'Slow moving';}

                $data = [
                    'sp_id' => $sparepart->sp_id,
                    'series' => $sparepart->sp_series,
                    'die_name' => $sparepart->sp_die_name,
                    'itemno' => $sparepart->sp_itemno,
                    'description' => $sparepart->sp_desc,
                    'supplier' => $sparepart->supplier_name,
                    'unit_price' => $sparepart->sp_unitprice,
                    'opening_stock' => $sparepart->sp_opening_stock,
                    'opening_amount' => $sparepart->sp_opening_amount,
                    'stockin_qty' => $stockInQty,
                    'stockin_amount' => number_format($stockInQty * $sparepart->sp_unitprice, 2),
                    'stockout_qty' => $stockOutQty,
                    'stockout_amount' => number_format($stockOutQty * $sparepart->sp_unitprice, 2),
                    'closing_qty' => $closingQty,
                    'closing_amount' => number_format($closingQty * $sparepart->sp_unitprice),
                    'status' => $status,
                    'latest_stockin_date' => $latestStockin != null ? date_format(new DateTime($latestStockin), 'd/m/y') : '-',
                    'latest_stockout_date' => $latestStockOut != null ? date_format(new DateTime($latestStockOut), 'd/m/y') : '-',
                    'month' => $month,
                    'rank' => $rank
                ];

                // $total = $closingQty * $sparepart->sp_unitprice;

                //Put data in each row excel
                $sheet->setCellValue('A'.$column, $no);
                $sheet->setCellValue('B'.$column, $data['series']);
                $sheet->setCellValue('C'.$column, $data['die_name']);
                $sheet->setCellValue('D'.$column, $data['itemno']);
                $sheet->setCellValue('E'.$column, $data['description']);
                $sheet->setCellValue('F'.$column, $data['supplier']);
                $sheet->setCellValue('G'.$column, 'RM '.$data['unit_price']);
                $sheet->setCellValue('H'.$column, $data['opening_stock']);
                $sheet->setCellValue('I'.$column, 'RM '.$data['opening_amount']);
                $sheet->setCellValue('J'.$column, $data['stockin_qty']);
                $sheet->setCellValue('K'.$column, 'RM '.$data['stockin_amount']);
                $sheet->setCellValue('L'.$column, $data['stockout_qty']);
                $sheet->setCellValue('M'.$column, 'RM '.$data['stockout_amount']);
                $sheet->setCellValue('N'.$column, $data['closing_qty']);
                $sheet->setCellValue('O'.$column, 'RM '.$data['closing_amount']);
                $sheet->setCellValue('P'.$column, $data['status']);
                $sheet->setCellValue('Q'.$column, $data['latest_stockin_date']);
                $sheet->setCellValue('R'.$column, $data['latest_stockout_date']);
                $sheet->setCellValue('S'.$column, $data['month']);
                $sheet->setCellValue('T'.$column, $data['rank']);

                //Centered all column in excel
                $centeredColumns = ['A','B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T'];
                foreach ($centeredColumns as $columns) {
                    $sheet->getStyle($columns)->getAlignment()->setHorizontal(\PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER);
                }

                //Colored column in excel
                $sheet->getStyle('G'.$column)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('FFCCFF');
                $sheet->getStyle('H'.$column)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('FFFF99');
                $sheet->getStyle('J'.$column)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('FFFF99');
                $sheet->getStyle('L'.$column)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('FFFF99');
                $sheet->getStyle('N'.$column)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('CCFFCC');
                if($status == 'Non Active'){
                    $sheet->getStyle('P'.$column)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('FF0000');
                }elseif($status == 'Active'){
                    $sheet->getStyle('P'.$column)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('00B050');
                }elseif($status == 'Slow moving'){
                    $sheet->getStyle('P'.$column)->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)->getStartColor()->setARGB('FFFF00');
                }

                //Set currency format
                $currencyFormat = 'RM #,##0.00';
                $sheet->getStyle('G'.$column)->getNumberFormat()->setFormatCode($currencyFormat);
                $sheet->getStyle('I'.$column)->getNumberFormat()->setFormatCode($currencyFormat);
                $sheet->getStyle('K'.$column)->getNumberFormat()->setFormatCode($currencyFormat);
                $sheet->getStyle('M'.$column)->getNumberFormat()->setFormatCode($currencyFormat);
                $sheet->getStyle('O'.$column)->getNumberFormat()->setFormatCode($currencyFormat);

                $no++;
                $column++;
                // $sum += $total;
            }

            //Total Expenses
            $sheet->setCellValue('T4', 'RM '.$total_expenses);

            //Border in excel
            $borderStyle = \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN;
            $sheet->getStyle('A10:T'.($column - 1))->getBorders()->getAllBorders()->setBorderStyle($borderStyle);

            //Save and download
            $path = 'report/SparePartMulti_report_'.$departments.'.xlsx';
            $filePath = public_path($path);
            $filePaths[] = $filePath;

            $filenameExcel = 'SparePartMulti_report.xlsx';

            $writer = new Xlsx($spreadsheet);
            $writer->save(public_path($path));
        }


        $existingZipFilePath = public_path('SparePartMulti_reports.zip');
        if (file_exists($existingZipFilePath)) {
            unlink($existingZipFilePath);
        }

            if($type == 'pdf'){
                $pdfFiles = [];

                foreach ($filePaths as $filePath) {
                    $getting_file = $filePath;
                    $reader = IOFactory::createReader('Xlsx');
                    $spreadsheet2 = $reader->load($getting_file);
                    $spreadsheet2->getActiveSheet()->getPageSetup()->setOrientation(PageSetup::ORIENTATION_LANDSCAPE);
                    $writers = IOFactory::createWriter($spreadsheet2, 'Mpdf');
                    // $pdf_file = 'SparePartMulti_report.pdf';
                    $pdf_file = basename($filePath, '.xlsx') . '.pdf';
                    $pdfFiles[] = $pdf_file;

                    $writers->save($pdf_file);
                }

                // If there are multiple PDF files, zip them
                if (count($pdfFiles) > 1) {
                    $zip = new ZipArchive();
                    $zipFileName = 'SparePartMulti_reports.zip';
                    $zipFilePath = public_path($zipFileName);
                    if ($zip->open($zipFilePath, ZipArchive::CREATE) === TRUE) {
                        foreach ($pdfFiles as $pdfFile) {
                            $zip->addFile($pdfFile, basename($pdfFile));
                        }
                        $zip->close();
                        return response()->download($zipFilePath, $zipFileName);
                    }
                } else {
                    // If there's only one PDF file, directly download it
                    return response()->download(public_path($pdfFiles[0]), basename($pdfFiles[0]));
                }

                // return response()->download(public_path('SparePartMulti_report.pdf'), 'SparePartMulti_report.pdf');

                // $spreadsheet = IOFactory::load(public_path($path));
                // $excelTempPath = tempnam(sys_get_temp_dir(), 'excel') . '.xlsx';
                // $excelWriter = new Xlsx($spreadsheet);
                // $excelWriter->save($excelTempPath);
                // $dompdf = new PDF();
                // $html = file_get_contents($excelTempPath);
                // $dompdf->loadHtml($html);
                // $dompdf->setPaper('A4', 'landscape');
                // $dompdf->render();
                // $pdfPath = 'report/SparePartReport.pdf';
                // $pdfFilePath = public_path($pdfPath);
                // file_put_contents($pdfFilePath, $dompdf->output());
                // unlink($excelTempPath);
                // $filenamePDF = 'Spare_PartMulti.pdf';
                // return response()->download($pdfFilePath, $filenamePDF);

            }else{
                // foreach ($filePaths as $filePath) {
                //     return response()->download($filePath, basename($filePath));
                // }

                if (count($filePaths) > 1) {
                    // Zip all Excel files
                    $zip = new ZipArchive();
                    $zipFileName = 'SparePartMulti_reports.zip';
                    $zipFilePath = public_path($zipFileName);
                    if ($zip->open($zipFilePath, ZipArchive::CREATE) === TRUE) {
                        foreach ($filePaths as $filePath) {
                            $zip->addFile($filePath, basename($filePath));
                        }
                        $zip->close();
                        return response()->download($zipFilePath, $zipFileName);
                    }
                } else {
                    // If there's only one Excel file, directly download it
                    foreach ($filePaths as $filePath) {
                        return response()->download($filePath, basename($filePath));
                    }
                }
            }

    }

    // public function downloadMultiExcel(Request $request)
    // {
    //     // Extend the time limit to allow for longer processing
    //     set_time_limit(300);

    //     // Retrieve request parameters
    //     $department = $request['department'];
    //     $date = $request['month'];
    //     $type = $request['type'];

    //     // Create a new Spreadsheet instance
    //     $spreadsheet = IOFactory::load(public_path('template/SparePartReport.xlsx'));
    //     $sheet = $spreadsheet->getActiveSheet();


    //     // Loop through each department
    //     foreach($department as $departments){
    //         //Get Budget
    //         $budget = Budgetting::where('budget_department', $departments)
    //                 ->where('budget_month', $date)
    //                 ->value('budget_budget');
    //         if($budget == null){
    //             return back()->with('month', 'Budget Month does not exist');
    //         }
    //         $budget_format = number_format($budget, 2);

    //         //Total Expenses in Stock In
    //         $year = date('Y', strtotime($date));
    //         $month = date('m', strtotime($date));
    //         $month_year = date('m/Y', strtotime($date));
    //         $total_expenses = StockInOut::whereRaw("YEAR(created_at) = ? AND MONTH(created_at) = ?", [$year, $month])->sum('si_total');
    //         $department_name = Department::where('department_id', $departments)->value('department_name');

    //         $sheet->setCellValue('T3', 'RM '.$budget_format);
    //         $sheet->setCellValue('T4', 'RM '.$total_expenses);
    //         $sheet->setCellValue('T5', $department_name);
    //         $sheet->setCellValue('T6', $month_year);

    //         // Fetch data in chunks for the current department
    //         SparePart::join('supplier', 'supplier_id','=','sp_supplier_id')
    //             ->where('sp_department', $departments)
    //             ->where('spare_part.is_deleted', 0)
    //             ->chunk(100, function ($spareparts) use ($sheet) {
    //                 // Process each chunk of data
    //                 $column = 10;
    //                 $no = 1;

    //                 foreach($spareparts as $sparepart){
    //                     // Your existing code for processing data and writing to the sheet goes here
    //                     $stockInQty = StockInOut::where('si_sp_id', $sparepart->sp_id)->where('si_type', 'Stock In')->count();
    //                     $stockOutQty = StockInOut::where('si_sp_id', $sparepart->sp_id)->where('si_type', 'Stock Out')->count();
    //                     $closingQty = $sparepart->sp_opening_stock + $stockInQty + $stockOutQty;
    //                     $latestStockin = StockInOut::where('si_sp_id', $sparepart->sp_id)->where('si_type', 'Stock In')->orderBy('created_at', 'DESC')->value('created_at');
    //                     $latestStockOut = StockInOut::where('si_sp_id', $sparepart->sp_id)->where('si_type', 'Stock Out')->orderBy('created_at', 'DESC')->value('created_at');
    //                     $rank = $sparepart->sp_rank;

    //                     $now = Carbon::now();
    //                     $now = new DateTime();
    //                     $latestStockinDate = new DateTime($latestStockin);
    //                     $interval = $latestStockinDate->diff($now);
    //                     $month = round($interval->days / 30.44);

    //                     //Month Rank
    //                     // if($month == 0.0){$rank = '-';}
    //                     // elseif($month <= 6){$rank = 'A';}
    //                     // elseif($month >= 7 && $month < 12){$rank = 'B';}
    //                     // elseif($month >= 13 && $month < 24){$rank = 'C';}
    //                     // elseif($month >= 25 && $month < 48){$rank = 'D';}
    //                     // elseif($month >= 49){$rank = 'E';}

    //                     //Status
    //                     if($sparepart->sp_status == 0){$status = 'Non Active';}
    //                     elseif($sparepart->sp_status == 1){$status = 'Active';}
    //                     elseif($sparepart->sp_status == 2){$status = 'Slow moving';}

    //                     $data = [
    //                         'sp_id' => $sparepart->sp_id,
    //                         'series' => $sparepart->sp_series,
    //                         'die_name' => $sparepart->sp_die_name,
    //                         'itemno' => $sparepart->sp_itemno,
    //                         'description' => $sparepart->sp_desc,
    //                         'supplier' => $sparepart->supplier_name,
    //                         'unit_price' => $sparepart->sp_unitprice,
    //                         'opening_stock' => $sparepart->sp_opening_stock,
    //                         'opening_amount' => $sparepart->sp_opening_amount,
    //                         'stockin_qty' => $stockInQty,
    //                         'stockin_amount' => number_format($stockInQty * $sparepart->sp_unitprice, 2),
    //                         'stockout_qty' => $stockOutQty,
    //                         'stockout_amount' => number_format($stockOutQty * $sparepart->sp_unitprice, 2),
    //                         'closing_qty' => $closingQty,
    //                         'closing_amount' => number_format($closingQty * $sparepart->sp_unitprice),
    //                         'status' => $status,
    //                         'latest_stockin_date' => $latestStockin != null ? date_format($latestStockin, 'd/m/y') : '-',
    //                         'latest_stockout_date' => $latestStockOut != null ? date_format($latestStockOut, 'd/m/y') : '-',
    //                         'month' => $month,
    //                         'rank' => $rank
    //                     ];

    //                     // Your existing code for writing data to the spreadsheet goes here
    //                     $sheet->setCellValue('A'.$column, $no);
    //                     $sheet->setCellValue('B'.$column, $data['series']);
    //                     $sheet->setCellValue('C'.$column, $data['die_name']);
    //                     $sheet->setCellValue('D'.$column, $data['itemno']);
    //                     $sheet->setCellValue('E'.$column, $data['description']);
    //                     $sheet->setCellValue('F'.$column, $data['supplier']);
    //                     $sheet->setCellValue('G'.$column, 'RM '.$data['unit_price']);
    //                     $sheet->setCellValue('H'.$column, $data['opening_stock']);
    //                     $sheet->setCellValue('I'.$column, 'RM '.$data['opening_amount']);
    //                     $sheet->setCellValue('J'.$column, $data['stockin_qty']);
    //                     $sheet->setCellValue('K'.$column, 'RM '.$data['stockin_amount']);
    //                     $sheet->setCellValue('L'.$column, $data['stockout_qty']);
    //                     $sheet->setCellValue('M'.$column, 'RM '.$data['stockout_amount']);
    //                     $sheet->setCellValue('N'.$column, $data['closing_qty']);
    //                     $sheet->setCellValue('O'.$column, 'RM '.$data['closing_amount']);
    //                     $sheet->setCellValue('P'.$column, $data['status']);
    //                     $sheet->setCellValue('Q'.$column, $data['latest_stockin_date']);
    //                     $sheet->setCellValue('R'.$column, $data['latest_stockout_date']);
    //                     $sheet->setCellValue('S'.$column, $data['month']);
    //                     $sheet->setCellValue('T'.$column, $data['rank']);

    //                     // Increment counters or other necessary operations
    //                     $column++;
    //                     $no++;
    //                 }
    //             });

    //         //Save and download
    //         $path = 'report/SparePartMulti_report_'.$departments.'.xlsx';
    //         $filePath = public_path($path);
    //         $filePaths[] = $filePath;

    //         $filenameExcel = 'SparePartMulti_report.xlsx';

    //         $writer = new Xlsx($spreadsheet);
    //         $writer->save(public_path($path));
    //     }

    //     $existingZipFilePath = public_path('SparePartMulti_reports.zip');
    //     if (file_exists($existingZipFilePath)) {
    //         unlink($existingZipFilePath);
    //     }

    //     if($type == 'pdf'){
    //         $pdfFiles = [];

    //         foreach ($filePaths as $filePath) {
    //             $getting_file = $filePath;
    //             $reader = IOFactory::createReader('Xlsx');
    //             $spreadsheet2 = $reader->load($getting_file);
    //             $spreadsheet2->getActiveSheet()->getPageSetup()->setOrientation(PageSetup::ORIENTATION_LANDSCAPE);
    //             $writers = IOFactory::createWriter($spreadsheet2, 'Mpdf');
    //             // $pdf_file = 'SparePartMulti_report.pdf';
    //             $pdf_file = basename($filePath, '.xlsx') . '.pdf';
    //             $pdfFiles[] = $pdf_file;

    //             $writers->save($pdf_file);
    //         }

    //         // If there are multiple PDF files, zip them
    //         if (count($pdfFiles) > 1) {
    //             $zip = new ZipArchive();
    //             $zipFileName = 'SparePartMulti_reports.zip';
    //             $zipFilePath = public_path($zipFileName);
    //             if ($zip->open($zipFilePath, ZipArchive::CREATE) === TRUE) {
    //                 foreach ($pdfFiles as $pdfFile) {
    //                     $zip->addFile($pdfFile, basename($pdfFile));
    //                 }
    //                 $zip->close();
    //                 return response()->download($zipFilePath, $zipFileName);
    //             }
    //         } else {
    //             // If there's only one PDF file, directly download it
    //             return response()->download(public_path($pdfFiles[0]), basename($pdfFiles[0]));
    //         }

    //         // return response()->download(public_path('SparePartMulti_report.pdf'), 'SparePartMulti_report.pdf');

    //         // $spreadsheet = IOFactory::load(public_path($path));
    //         // $excelTempPath = tempnam(sys_get_temp_dir(), 'excel') . '.xlsx';
    //         // $excelWriter = new Xlsx($spreadsheet);
    //         // $excelWriter->save($excelTempPath);
    //         // $dompdf = new PDF();
    //         // $html = file_get_contents($excelTempPath);
    //         // $dompdf->loadHtml($html);
    //         // $dompdf->setPaper('A4', 'landscape');
    //         // $dompdf->render();
    //         // $pdfPath = 'report/SparePartReport.pdf';
    //         // $pdfFilePath = public_path($pdfPath);
    //         // file_put_contents($pdfFilePath, $dompdf->output());
    //         // unlink($excelTempPath);
    //         // $filenamePDF = 'Spare_PartMulti.pdf';
    //         // return response()->download($pdfFilePath, $filenamePDF);

    //     }else{
    //         // foreach ($filePaths as $filePath) {
    //         //     return response()->download($filePath, basename($filePath));
    //         // }

    //         if (count($filePaths) > 1) {
    //             // Zip all Excel files
    //             $zip = new ZipArchive();
    //             $zipFileName = 'SparePartMulti_reports.zip';
    //             $zipFilePath = public_path($zipFileName);
    //             if ($zip->open($zipFilePath, ZipArchive::CREATE) === TRUE) {
    //                 foreach ($filePaths as $filePath) {
    //                     $zip->addFile($filePath, basename($filePath));
    //                 }
    //                 $zip->close();
    //                 return response()->download($zipFilePath, $zipFileName);
    //             }
    //         } else {
    //             // If there's only one Excel file, directly download it
    //             foreach ($filePaths as $filePath) {
    //                 return response()->download($filePath, basename($filePath));
    //             }
    //         }
    //     }
    // }

    public function showAddSingleStockIn($sp_id)
    {
        if(Auth::check()){

            $part = SparePart::join('supplier', 'supplier_id','=','sp_supplier_id')
                    ->where('spare_part.is_deleted', 0)
                    ->where('spare_part.sp_id', $sp_id)
                    ->get();
            $today = new DateTime(Carbon::now());
            $now = date_format($today, 'd/m/Y');

            $notification = Notifications::orderBy('notification_is_read', 'ASC')->get();
            $new_notification = Notifications::where('notification_is_read', 0)->count();

            return view('addsinglestockin')->with([
                'part' => $part,
                'now' => $now,
                'notification' => $notification,
                'new_notification' => $new_notification,
            ]);
        }
        return redirect()->to('login')->withSuccess('You are not allowed to access');
    }


}
